!**********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! *
! * SUBROUTINE: Fan Coil Unit     
! *
! * LANGUAGE:   FORTRAN 77
! *
! * PURPOSE:    Calculates supply air and fresh air flow rates, 
! *             return air pressure and fan power consumption
! *             and added heat to flow stream for fan coil units.
!***********************************************************************
! * INPUTS
! * ======
! *  1. WSA     : Supply air mass flow rate                        (kg/s)
! *  2. PO      : Outlet fan coil pressure                          (kPa)
! *  3. WRET    : Return air mass flow rate                        (kg/s)
! *  4. POUT    : Outside air intake pressure                       (kPa)
! *  5. COUT    : Outside air damper position 
! *               [(0=closed)<=  <=(1=open)]                          (-)
! *  6. N       : Rotational speed of the fan                     (rev/s)
! *
! * OUTPUTS
! * =======
! *  1. QA      : Fluid stream heat addition rate                    (kW)
! *  2. POWER   : Power consumption by fan                           (kW)
! *  3. WSA     : Supply air mass flow rate                        (kg/s)
! *  4. PRET    : Return air pressure                               (kPa)
! *  5. WOUT    : Outside air mass flow rate                       (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. K       : Number of fans                                      (-)
! *  2.         : coefficient of PN**0 in normalized head curve
! *  3.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  4.         : coefficient of PN**4 in normalized head curve
! *  5.         : coefficient of PN**0 in normalized efficiency curve
! *  6.         : coefficient of PN**1 in normalized efficiency curve
! * ...
! * 11.         : coefficient of PN**4 in normalized efficiency curve
! * 12. D       : fan diameter                                        (m)
! * 13. WNCRITLO: normalised flow at lower bound of valid region      (-)
! * 14. WNCRITUP: normalised flow at upper bound of valid region      (-)
! * 15. R       : resistance of fan coil components           (0.001/k.m)
! * 16. FAREAOUT: Face area of outside air damper                    (m2)
! * 17. FAREARET: Face area of return air damper                     (m2)
! *
! * SAVED
! * =====
! *  1.         : flag used to indicate initialization has been performed
! *               (used in HVACSIM+ only)(RELATED TO FAN SUBROUTINE)
! *  2. ALO/N**2: A coefficient in lower region, divided by N squared
! *               (RELATED TO FAN SUBROUTINE)
! *  3. RLO     : equivalent resistance in lower region
! *               (RELATED TO FAN SUBROUTINE)
! *  4. AUP/N**2: A coefficient in upper region, divided by N squared
! *               (RELATED TO FAN SUBROUTINE)
! *  5. RUP     : equivalent resistance in upper region
! *               (RELATED TO FAN SUBROUTINE)
!***********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes positive mixed flow rate and fixed
!                        blade angle, no inlet guide vanes for fan,
!                        all inefficiencies appear as heat in the fluid
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen   
!                        Drexel University 
!
!   DATE:                April, 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 355
!   FUNCTIONS  CALLED:   DPQUDLIN, RDAMPER, WQUDLIN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
!***********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * ROUT    : total resistance of outside air branch
! * RRET    : total resistance of return air branch
! * ROPENOUT: open resistance of outside air branch
! * ROPENRET: open resistance of return air branch 
! * DP      : pressure drop across outside air branch
! * PMIX    : mixed air pressure
! * CRET    : return air damper position
!***********************************************************************
        
	    subroutine type307(xin,yout,par_v,saved_v,iostat)
        
		use modsim_head
        implicit none
		integer,parameter                    :: ni=6,no=5,np=17,ns=5
        real,dimension(ni),intent(in)        :: xin
        real,dimension(no),intent(out)       :: yout                     
        real,dimension(np),intent(in)        :: par_v                    
        real,dimension(ns),intent(in out)    :: saved_v                  
        integer,dimension(no),intent(out)    :: iostat

		integer,parameter                    :: nifan=3,nofan=3,npfan=15,nsfan=5
        real,dimension(nifan)                :: xinfan
        real,dimension(nofan)                :: youtfan                     
        real,dimension(npfan)                :: par_vfan                    
        real,dimension(nsfan)                :: saved_vfan                  
        integer,dimension(nofan)             :: iostatfan

! **** Damper coefficients (ln(k)=a+b*theta) from legg

         real,dimension(0:1)   :: ALEGG=(/0.4877,-1.51/),&
                                  BLEGG=(/0.0715,0.0842/)

! **** Loss coefficient of fully open damper
!       
         real         :: KOPEN=2.90835

! **** Assumed fractional flow leakage of closed damper
!       
         real         :: FLEAK=1.0e-3

		
! **** Assume opposed or single blade damper
        integer      :: ipx=0 

        real         :: RHOA=1.2,CPA=1.0
		 
		real         :: WSA,PO,WRET,POUT,COUT,N,QA,POWER,PRET,WOUT,WOUT1,&
		                D,WNCRITLO,WNCRITUP,R,FAREAOUT,FAREARET,WFAN,&
						PI,CRET,PMIX,ROPENOUT,ROPENRET,ROUT,RRET,DP,&
						DPQUDLIN,RDAMPER,WQUDLIN,ALEGG1,BLEGG1

        integer      :: K,I


!*** Read in inputs used in fan subroutine
!*** Following fan laws in parallel the total mass flow rate is devided between fans  
        K        = par_v(1)
		WSA      = xin(1)
	    WFAN     = WSA/K
        xinfan(1)= WFAN
        xinfan(2)= xin(2)
	    xinfan(3)= xin(6)

!*** Read in parameters used in fan subroutine
        
	    par_vfan(1) = par_v(2)
	    par_vfan(2) = par_v(3)
	    par_vfan(3) = par_v(4)
	    par_vfan(4) = par_v(5)
	    par_vfan(5) = par_v(6)
	    par_vfan(6) = par_v(7)
	    par_vfan(7) = par_v(8)
	    par_vfan(8) = par_v(9)
	    par_vfan(9) = par_v(10)
	    par_vfan(10)= par_v(11)
	    par_vfan(11)= par_v(12)
        par_vfan(12)= 1   !MODE    : fluid: 1 = air, any other value = water
	    par_vfan(13)= par_v(13)
	    par_vfan(14)= par_v(14)
        par_vfan(15)= 1   !LOCATION: 1-SA, 2-RA 

!*** Call fan subroutine to calculate inlet pressure, power and added heat to fluid stream

        CALL TYPE355(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)

        PI   = youtfan(1)
	    QA   = youtfan(2)
	    POWER= youtfan(3)
!       QA   = K*QA
!		POWER= K*POWER
		   
!*** Read in parameters for mixing box and coil resistance

        R    = par_v(15)
        IF (R<=0.0) STOP 'TYPE307: R NOT GREATER THAN 0'

!*** Fault flag on*********************************************
!		IF (Heatingfoul.EQ.1) THEN
!       IF (heatcoilfault(2).EQ.1) THEN
		IF (coilfault(3).EQ.1.OR.coilfault(4).EQ.1) THEN

!*** Heating coil air side fouling let air resistance increased
!			R = Vheatingfoul1*par_v(15)
!           R = vheatcoilfault(2)*par_v(15)
			R = vcoilfault(2)*par_v(15)
		ENDIF
	    
!		IF (Coolingfoul.EQ.1) THEN
!       IF (coolcoilfault(2).EQ.1) THEN

!*** Cooling coil air side fouling let air resistance increased
!			R = Vcoolingfoul1*par_v(15)
!           R = vcoolcoilfault(2)*par_v(15)
!		ENDIF

        IF (Filter.EQ.1) THEN

!*** Filter blockage let air resistance increased
			R = Vfilter*par_v(15)
		
		ENDIF
!***  Fault flag off********************************************
	          
	    FAREAOUT= par_v(16)
	    
		IF (FAREAOUT<=0.0) STOP 'TYPE307: FAREAOUT NOT GREATER THAN 0'

!*** Fault flag on**********************************************
        
		IF (Economizer.EQ.1) THEN

!*** Opening blockage let outdoor air inlet area decreased
		    FAREAOUT = Veconomizer*par_v(16)
		
		ENDIF
!*** Fault flag off**********************************************

        FAREARET= par_v(17)
	    IF (FAREARET<=0.0) STOP 'TYPE307: FAREARET NOT GREATER THAN 0'
        
!*** Read in inputs
        WSA  = xin(1)
        WRET = xin(3)
        POUT = xin(4)
	    COUT = xin(5)
!*** Fault flag on************************************************                       
		IF (OAdamp.EQ.1) THEN
!*** OA damper is stuck
			COUT = VOAdamp
        ELSEIF (OAdamp.EQ.2) THEN
		       IF (COUT.EQ.0) THEN
!*** OA damper is leaking
			       COUT = VOAdamp
			   ENDIF
		ENDIF
!*** Fault flag off************************************************  

 	    CRET =1-COUT 
!*** Mixed air pressure is calculated      	
	    PMIX=PI+DPQUDLIN(R,WSA)
	  
!**** Fresh air and return air damper resistances
!**** Resistance when fully open
        ROPENOUT = 1.0E-3*KOPEN/(2.0*RHOA*FAREAOUT*FAREAOUT)
        ROPENRET = 1.0E-3*KOPEN/(2.0*RHOA*FAREARET*FAREARET) 
! **** Resistance at position damppos
        ALEGG1 = ALEGG(0)                 
        BLEGG1 = BLEGG(0)                 
        ROUT = RDAMPER(COUT,ROPENOUT,FLEAK,FAREAOUT,ALEGG1,BLEGG1,ipx) 
	  	
        RRET = RDAMPER(CRET,ROPENRET,FLEAK,FAREARET,ALEGG1,BLEGG1,ipx) 
  
!*** Calculate outside air mass flow rate
        DP=POUT-PMIX
        WOUT1=WQUDLIN(ROUT,DP)

!*** Fault flag on*************************************************
		IF (Economizer.EQ.0) THEN
!*** No fault for economizer
			WOUT = WOUT1
				
		ELSEIF (Economizer.EQ.2) THEN
			IF (COUT.EQ.0) THEN
!*** OA damper is leaking
				WOUT = Veconomizer
				
			ELSE
				WOUT = WOUT1
			ENDIF
		ENDIF
!*** Fault flag off*************************************************

!*** Calculate return air pressure

       PRET=PMIX+DPQUDLIN(RRET,WRET)
!*** Calculate mixed air flow rate as sum of outside and recirc flow rates
       WSA=WOUT+WRET

!*** Assign outputs
       yout(1) = QA
       yout(2) = POWER
       yout(3) = WSA
       yout(4) = PRET
       yout(5) = WOUT
        
!	  SAVED(1)=SAVEDFAN(1)
!	  SAVED(2)=SAVEDFAN(2)
!	  SAVED(3)=SAVEDFAN(3)
!	  SAVED(4)=SAVEDFAN(4)
!	  SAVED(5)=SAVEDFAN(5)
        
!*** Allow freezing of algebraic variables
       do  I=1,NO
            IOSTAT(I)=1
       enddo
!*** Return
       RETURN
        
       end subroutine type307
!**********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! * 
! * SUBROUTINE: ERS Fan Coil Unit 
! * LANGUAGE:   FORTRAN 77
! * 
! * PURPOSE:    Calculates supply air and fresh air flow rates, 
! *             fan outlet pressure and fan power consumption
! *             and added heat to flow stream for fan coil units
! *             controlled when the percentage of return air 
! *             vs. fresh air damper position is given.
! *              
!***********************************************************************
! * INPUTS
! * ======
! *  1. WRET    : Return air mass flow rate                        (kg/s) 
! *  2. POUT    : Outside air intake pressure                       (kPa)
! *  3. COUT    : Outside air damper position (0=closed, 1=open)      (-)
! *  4. N       : Rotational speed of the fan                     (rev/s)
! *   
! * OUTPUTS
! * =======
! *  1. QA      : Fluid stream heat addition rate                    (kW)
! *  2. POWER   : Power consumption                                  (kW)
! *  3. WSA     : Supply air mass flow rate                        (kg/s)
! *  4. WOUT    : Outside air mass flow rate                       (kg/s)
! *  5. PO      : Outlet fan coil pressure                          (kPa)
! * 
! * PARAMETERS
! * ==========
! *  1.         : coefficient of PN**0 in normalized head curve
! *  2.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  5.         : coefficient of PN**4 in normalized head curve
! *  6.         : coefficient of PN**0 in normalized efficiency curve
! *  7.         : coefficient of PN**1 in normalized efficiency curve
! * ...
! * 10.         : coefficient of PN**4 in normalized efficiency curve
! * 11.         : coefficient of PN**0 in percentage return air curve
! * 12.         : coefficient of PN**1 in percentage return air curve
! * ...
! * 14.         : coefficient of PN**3 in percentage return air curve
! * 15. K       : number of fans in fan coil unit                    (-)
! * 16. D       : fan diameter                                       (m)
! * 17. WNCRITLO: normalised flow at lower bound of valid region     (-)
! * 18. WNCRITUP: normalised flow at upper bound of valid region     (-)
! * 19. R       : resistance of fan coil components          (0.001/k.m)
! * 20. FAREAOUT: face area of outside air damper                   (m2)
! * 
! * SAVED
! * =====
! *  1.         : flag used to indicate initialization has been performed
! *               (used in HVACSIM+ only)(RELATED TO FAN SUBROUTINE)
! *  2. ALO/N**2: A coefficient in lower region, divided by N squared
! *               (RELATED TO FAN SUBROUTINE)
! *  3. RLO     : equivalent resistance in lower region
! *               (RELATED TO FAN SUBROUTINE)
! *  4. AUP/N**2: A coefficient in upper region, divided by N squared
! *               (RELATED TO FAN SUBROUTINE)
! *  5. RUP     : equivalent resistance in upper region
! *               (RELATED TO FAN SUBROUTINE)
!***********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes positive mixed flow rate and fixed
!                        blade angle, no inlet guide vanes for fan,
!                        all inefficiencies appear as heat in the fluid
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen   
!                        Drexel University 
!
!   DATE:                April, 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 355
!   FUNCTIONS  CALLED:   DPQUDLIN, RDAMPER, WQUDLIN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
!***********************************************************************
!  
!  INTERNAL VARIABLES
!  ==================
! * ROUT    : total resistance of outside air branch
! * RRET    : total resistance of return air branch
! * ROPENOUT: open resistance of outside air branch
! * ROPENRET: open resistance of return air branch 
! * DP      : pressure drop across outside air branch
! * PMIX    : mixed air pressure
! * CRET    : return air damper position
! * 
!***********************************************************************
      
	   subroutine type309(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                    :: ni=4,no=5,np=20,ns=5
        real,dimension(ni),intent(in)        :: xin
        real,dimension(no),intent(out)       :: yout                     
        real,dimension(np),intent(in)        :: par_v                    
        real,dimension(ns),intent(in out)    :: saved_v                  
        integer,dimension(no),intent(out)    :: iostat

! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: ALEGG=(/-1.51,-1.51/),&
                                 BLEGG=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: KOPEN=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: FLEAK=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0 

        real         :: RHO=1.2,CPA=1.0
		 
		real         :: WRET,POUT,COUT,N,QA,POWER,WSA,WOUT,&
		                WOUT1,PO,D,WNCRITLO,WNCRITUP,R,FAREAOUT   

		real         :: ROPENOUT,ROUT,PMIX,DPQUDLIN,PI,WFAN,&
		                DWN,WN,ANLO,RNLO,ANUP,RNUP,ROFF,PDROP,&
                        ALO,RLO,PRISE,EFF,AUP,RUP,PN,Z,FUNP,&
						DFUNP,FUNN,FUNWRET,ALEGG1,BLEGG1,RDAMPER
 
		integer      :: K,I


!*** Fan functions
!*** "Head curve" - normalized pressure rise as a function of normalized flow
        FUNP(Z)  = par_v(1)+Z*(par_v(2)+Z*(par_v(3)+Z*(par_v(4)+Z*par_v(5))))
!*** Derivative of head curve
        DFUNP(Z) = par_v(2)+Z*(2.0*par_v(3)+Z*(3.0*par_v(4)+Z*4.0*par_v(5)))
!*** Efficiency curve
        FUNN(Z)  = par_v(6)+Z*(par_v(7)+Z*(par_v(8)+Z*(par_v(9)+Z*par_v(10))))

!*** Return air proportion function
!*** Percentage of return air flow rate in supply air as a function of fresh air damper position (COUT) 
	    FUNWRET(Z)  = par_v(11)+Z*(par_v(12)+Z*(par_v(13)+Z*par_v(14)))  

!*** Read in inputs 
       
	    WRET = xin(1)
	    POUT = xin(2)
	    COUT = xin(3)
!*** Fault flag on************************************************                       
!		IF (OAdamp.EQ.1) THEN

!*** OA damper is stuck
!			COUT = VOAdamp

!		ENDIF
!*** Fault flag off************************************************  
	    N    = xin(4)

!*** Read in parameters related to fan but not used in head curve and efficiency functions
        K        = par_v(15)
	    D        = par_v(16)
        WNCRITLO = par_v(17)
        WNCRITUP = par_v(18)
	  
!*** Read in parameters for mixing box and coil resistance 
        R        = par_v(19) 
	    IF (R<=0.0) STOP 'TYPE309: R NOT GREATER THAN 0'

!*** Fault flag on*********************************************
!		IF (Heatingfoul.EQ.1) THEN

!*** Heating coil air side fouling let air resistance increased
!			R = Vheatingfoul1*par_v(19)

!		ENDIF
	    
!		IF (Coolingfoul.EQ.1) THEN

!*** Cooling coil air side fouling let air resistance increased
!			R = Vcoolingfoul1*par_v(19)
		
!		ENDIF

!        IF (Filter.EQ.1) THEN

!*** Filter blockage let air resistance increased
!			R = Vfilter*par_v(19)
		
!		ENDIF
!***  Fault flag off********************************************
	          		       
	  
	    FAREAOUT = par_v(20)
	    IF (FAREAOUT<=0.0) STOP 'TYPE309: FAREAOUT NOT GREATER THAN 0'

!*** Fault flag on**********************************************
        
!		IF (Economizer.EQ.1) THEN

!*** Opening blockage let outdoor air inlet area decreased
!		    FAREAOUT = Veconomizer*par_v(20)
		
!		ENDIF
!*** Fault flag off**********************************************
                
	    WSA = 100*WRET/FUNWRET(COUT)

!*** Calculate fresh air folwrate 	  
	    WOUT1=WSA-WRET
!*** Fault flag on*************************************************
!		IF (Economizer.EQ.0) THEN
!*** No fault for economizer
!			WOUT = WOUT1
!				
!		ELSEIF (Economizer.EQ.2) THEN
!			IF (COUT.EQ.0) THEN
!*** OA damper is leaking
!				WOUT = Veconomizer
!				
!			ELSE
!				WOUT = WOUT1
!			ENDIF
!		ENDIF
!*** Fault flag off*************************************************

!*** Calculate resistance of fresh air damper
!*** Resistance when fully open
        ROPENOUT = 1.0E-3*KOPEN/(2.0*RHO*FAREAOUT*FAREAOUT)
!**** Resistance at position damppos
        ALEGG1 = ALEGG(0)                 
        BLEGG1 = BLEGG(0)                 
        ROUT = RDAMPER(COUT,ROPENOUT,FLEAK,FAREAOUT,ALEGG1,BLEGG1,ipx) 
	  	
!*** Calculate mixed air pressure
        PMIX=POUT-DPQUDLIN(ROUT,WOUT)
         
!*** Calculate fan inlet pressure      	
	    PI=PMIX-DPQUDLIN(R,WSA)
	  
!*** Fan outlet pressure calculation
!*** Calculate fan flow rate (parallel fans)
        WFAN=WSA/K
!*** Limit rotation speed to positive values
        N = MAX(0.0,N)
!*** "One-off" calculations
!*** Test for first call of simulation, assuming SAVED array is
!*** initialized to zero
        IF (INIT==0 .AND. saved_v(1)==0.0) THEN
            saved_v(1) = 1.
!*** Check head curve is monotonically decreasing
!*** Test gradient at 50 points across user-defined range
            DWN = (WNCRITUP-WNCRITLO)/50.
            WN  = WNCRITLO
            DO I=0,50
                IF (DFUNP(WN)>0.0) THEN
                    WRITE(*,*)&
                    'TYPE 309: fan head curve slope positive at WN =',WN
                ENDIF
                WN = WN+DWN
            ENDDO
!*** Determine coefficents of quadratic fits outside valid region
            ANLO = FUNP(WNCRITLO)-WNCRITLO*DFUNP(WNCRITLO)/2.0
!            IF (ANLO.LE.0.0) STOP 'TYPE350: ANLO<=0'
            RNLO = -DFUNP(WNCRITLO)/(2.0*ABS(WNCRITLO))
!            IF (RNLO.LE.0.0) STOP 'TYPE350: RNLO<=0'
            ANUP = FUNP(WNCRITUP)-WNCRITUP*DFUNP(WNCRITUP)/2.0
!            IF (ANUP.LE.0.0) STOP 'TYPE350: ANUP<=0'
            RNUP = -DFUNP(WNCRITUP)/(2.0*ABS(WNCRITUP))
!            IF (RNUP.LE.0.0) STOP 'TYPE350: RNUP<=0'
!*** Store partly or completely un-normalized coefficients
            saved_v(2) = ANLO*0.001*RHO*D*D
            saved_v(3) = RNLO*0.001/(RHO*D*D*D*D)
            saved_v(4) = ANUP*0.001*RHO*D*D
            saved_v(5) = RNUP*0.001/(RHO*D*D*D*D)
        ENDIF
!*** Calculate pressure rise, fluid heating rate and power consumption
        IF (N<=0.0) THEN
!*** Fan/pump off
!*** Calculate the pressure drop assuming the resistance is equal to
!*** the equivalent resistance in the upper region
            ROFF  = saved_v(5)
            PDROP = DPQUDLIN(ROFF,WFAN)
            PO    = PI-PDROP
            POWER = 0.0
        ELSE
!*** Fan/pump on       
!*** Normalized flow rate
            WN = WFAN/(RHO*D*D*D*N)
            IF (WN<WNCRITLO) THEN
!*** Below valid region
!*** Calculate pressure rise from quadratic extrapolation of head curve
                ALO   = saved_v(2)*N*N
                RLO   = saved_v(3)
                PRISE = ALO-DPQUDLIN(RLO,WFAN)
                PO    = PI+PRISE
                EFF   = FUNN(WNCRITLO)
            ELSEIF (WN>WNCRITUP) THEN
!*** Above valid region
!*** Calculate pressure rise from quadratic extrapolation of head curve
                AUP   = saved_v(4)*N*N
                RUP   = saved_v(5)
                PRISE = AUP-DPQUDLIN(RUP,WFAN)
                PO    = PI+PRISE
                EFF   = FUNN(WNCRITUP)
            ELSE
!*** In valid range
!*** Calculate pressure rise from head curve
                PN    = FUNP(WN)
                PRISE = 0.001*PN*RHO*D*D*N*N
                PO    = PI+PRISE
                EFF   = FUNN(WN)
            ENDIF
!*** Power consumption - only correct within range of validity of
!*** efficiency polynomial
            IF (WFAN>0.0.AND.(PO-PI)>0.0) THEN
                POWER = (PO-PI)*WFAN/(RHO*EFF)
            ELSE
                POWER = 0.0
            ENDIF
        ENDIF
!*** Rate at which heat is added to air stream
!       POWER= K*POWER
        QA   = POWER
!*****Fault Flag on**************************************
!	    IF (sfan.EQ.0) THEN
!***'SF is fault free'
!            yout(5) = PO
			
!		ELSEIF (sfan.EQ.2) THEN
!*** 'SF has complete failure'
!			PO=PI
!			QA=0
!			POWER=0

!		ENDIF
!*****Fault Flag off************************************* 
!*** Assign outputs

        yout(1) = QA
        yout(2) = POWER
        yout(3) = WSA
        yout(4) = WOUT
        yout(5) = PO       
!*** Allow freezing of algebraic variables
        DO  I=1,NO
            IOSTAT(I)=1
        ENDDO
!*** Return
        RETURN
        
        END SUBROUTINE type309
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a three port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                     (C)
! *  2. G(1)      : Return air humidity ratio                       (kg/kg)
! *  3. W(1)      : Return air mass flow rate                        (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                      (C)
! *  5. G(2)      : Fresh air humidity ratio                        (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                         (kg/s)
! *  7. QA        : Fan heat addition rate to fluid stream             (kW)
! *  8. TWICLG    : Cooling coil inlet water temperature                (C)
! *  9. WWPRIMCLG : Cooling coil primary circuit water mass flow rate(kg/s)
! * 10. PWOCLG    : Cooling coil outlet water pressure                (kPa)
! * 11. VSTEMCLG  : Cooling coil valve stem position                    (-)
! * 12. TWIHTG    : Heating coil inlet water temperature                (C)
! * 13. WWPRIMHTG : Heating coil primary circuit water mass flow rate(kg/s)
! * 14. PWOHTG    : Heating coil outlet water pressure                (kPa)
! * 15. VSTEMHTG  : Heating coil valve stem position                    (-)
! * 16. TSDYN     : Effective coil surface temperature                  (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA     : Mixed air dry bulb temperature                        (C)
! *  2. GMA     : Mixed air humidity ratio                          (kg/kg)
! *  3. TS      : Effective coil surface temperature                    (C)
! *  4. TAO     : Fan coil outlet air dry bulb temperature              (C)
! *  5. GO      : Fan coil outlet air humidity ratio                (kg/kg)
! *  6. TWO     : Coil outlet water temperature                         (C)
! *  7. PWI     : Coil inlet water pressure                           (kPa)
! *  8. WW      : Water mass flow rate though coil                   (kg/s)
! *  9. TRET    : Mixed return water temperature                        (C)
! * 10. QA      : Total heat transfer from the coil to the air         (kW)
! * 11. SHR     : Sensible heat ratio of the coil                       (-)
! * 12. EFFECT  : Coil effectiveness                                    (-)
! * 13. BF      : Coil by-pass factor                                   (-)
! * 14. HO      : Fan coil outlet air specific enthalpy             (kJ/kg)
! * 15. RHO     : Fan Coil outlet air relative humidity                 (-)
! * 16. TWBO    : Fan Coil outlet air wet-bulb temperature              (C)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil            (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil            (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil  (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil            (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil            (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil  (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                            (m)
! * 12. HCOIL      : height of finned section                           (m)
! * 13. WCOIL      : width of finned section                            (m)
! * 14. DOTUBE     : tube outside diameter                              (m)
! * 15. THITUBE    : tube wall thickness                                (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 17. SPAFIN     : fin spacing (pitch)                                (m)
! * 18. THIFIN     : fin thickness                                      (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance        (0.001 kg.m)
! * 21. FRWBYPASCLG: cooling coil by-pass water flow resistance(0.001 kg.m)
! * 22. IVALTYPECLG: cooling coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 23. KVCLG      : cooling coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 24. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 25. SVCLG      : cooling coil valve rangability                     (-)
! * 26. CLCLG      : cooling coil valve leakage (fractional flow)       (-)
! * 27. FRWHTGCOIL : heating coil water flow resistance        (0.001 kg.m)
! * 28. FRWBYPASHTG: heating coil by-pass water flow resistance(0.001 kg.m)
! * 29. IVALTYPEHTG: heating coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 30. KVHTG      : heating coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 31. EQPCHARHTG : heating coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 32. SVHTG      : heating coil valve rangability                     (-)
! * 33. CLHTG      : heating coil valve leakage (fractional flow)       (-)
! * 34. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type310(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=2,&
                                             ni=16,no=16,np=35,&
                                             ns=1+ndiffeq*4
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

		integer,parameter                 :: nicoil=9,nocoil=15,npcoil=23,nscoil=7
        real,dimension(nicoil)            :: xincoil
        real,dimension(nocoil)            :: youtcoil                     
        real,dimension(npcoil)            :: par_vcoil                    
        real,dimension(nscoil)            :: saved_vcoil                  
        integer,dimension(nocoil)         :: iostatcoil

  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
         real                  :: QA,TWICLG,&
		                          WWPRIMCLG,PWOCLG,VSTEMCLG,TWIHTG,WWPRIMHTG,&
								  PWOHTG,VSTEMHTG,TSDYN,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAI,GI,GSAT 
	   
	    real                   :: PATMPA=101325.
		
!*** Read in mixing box inputs

        xinmb(1) = xin(1)
        xinmb(2) = xin(2)
        xinmb(3) = xin(3)
        xinmb(4) = xin(4)
        xinmb(5) = xin(5)
        xinmb(6) = xin(6)

!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
        xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= xin(7)
!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
	    par_vfan(2)=par_v(34)

!*** Call fan subroutine
        CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	  
!*** Read in cooling/heating coil inputs

        xincoil(1)= youtfan(1)
        TAI       = youtfan(1)
	    GI        = youtmb(2)
!*** Limit humidity ratio to saturated value
        GSAT      = FWPHI(TAI,100.0,PATMPA)
        GI        = MIN(GI,GSAT)
        xincoil(2)= GI
        xincoil(3)= 0
	    xincoil(4)= youtmb(3)
	    TWICLG    = xin(8)
        WWPRIMCLG = xin(9)
	    PWOCLG    = xin(10)
        VSTEMCLG  = xin(11)
	    TWIHTG    = xin(12)
	    WWPRIMHTG = xin(13)
        PWOHTG    = xin(14)
	    VSTEMHTG  = xin(15)
	    xincoil(9)= xin(16)

!*** Read in cooling/heating coil parameters
        par_vcoil(1) = par_v(1)
	    par_vcoil(2) = par_v(2)
	    par_vcoil(3) = par_v(3)
	    NROWCLG      = par_v(4)
	    NTPRCLG      = par_v(5)
	    NCIRCLG      = par_v(6)
	    LCOILCLG     = par_v(7)
	    NROWHTG      = par_v(8)
	    NTPRHTG      = par_v(9)
	    NCIRHTG      = par_v(10)
	    LCOILHTG     = par_v(11)
	    par_vcoil(8) = par_v(12)
	    par_vcoil(9) = par_v(13)
	    par_vcoil(10)= par_v(14)
	    par_vcoil(11)= par_v(15)
	    par_vcoil(12)= par_v(16)
	    par_vcoil(13)= par_v(17)
	    par_vcoil(14)= par_v(18)
	    par_vcoil(15)= par_v(19)
	    par_vcoil(16)= 0
	    FRWCLGCOIL = par_v(20)
	    FRWBYPASCLG= par_v(21)
	    IVALTYPECLG= par_v(22)
	    KVCLG      = par_v(23)
	    EQPCHARCLG = par_v(24)
	    SVCLG      = par_v(25)
	    CLCLG      = par_v(26)
	    FRWHTGCOIL = par_v(27)
	    FRWBYPASHTG= par_v(28)
	    IVALTYPEHTG= par_v(29)
	    KVHTG      = par_v(30)
	    EQPCHARHTG = par_v(31)
	    SVHTG      = par_v(32)
	    CLHTG      = par_v(33)
        
	    IF ((VSTEMCLG.NE.0.0).AND.(VSTEMHTG.NE.0.0)) THEN
	     WRITE(*,*)'TYPE310: Fan coil unit is in both heating and cooling mode!'
         STOP
        ENDIF
        	  
	    IF (VSTEMCLG.NE.0.0) THEN   
!*** Read in cooling coil inputs
            xincoil(5) = TWICLG
            xincoil(6) = WWPRIMCLG
            xincoil(7) = PWOCLG 
	        xincoil(8) = VSTEMCLG
!*** Read in cooling coil parameters
            par_vcoil(4) = NROWCLG
	        par_vcoil(5) = NTPRCLG 
	        par_vcoil(6) = NCIRCLG
	        par_vcoil(7) = LCOILCLG
	        par_vcoil(17)= FRWCLGCOIL
	        par_vcoil(18)= FRWBYPASCLG 
	        par_vcoil(19)= IVALTYPECLG
	        par_vcoil(20)= KVCLG
	        par_vcoil(21)= EQPCHARCLG
	        par_vcoil(22)= SVCLG 
	        par_vcoil(23)= CLCLG
!*** Call cooling coil subroutine  
            CALL TYPE523(xincoil,youtcoil,par_vcoil,saved_vcoil,iostatcoil)
		
!*** Assign output values
            yout(1)  = youtmb(1)
	        yout(2)  = youtmb(2)
            yout(3)  = youtcoil(1)
            yout(4)  = youtcoil(2)
            yout(5)  = youtcoil(3)
            yout(6)  = youtcoil(5)
            yout(7)  = youtcoil(6)
            yout(8)  = youtcoil(7)
            yout(9)  = youtcoil(8)
            yout(10) = youtcoil(9)
            yout(11) = youtcoil(10)
	        yout(12) = youtcoil(11)
	        yout(13) = youtcoil(12)
	        yout(14) = youtcoil(13)
	        yout(15) = youtcoil(14)
	        yout(16) = youtcoil(15)
	                
	    ELSEIF (VSTEMHTG.NE.0.0) THEN
!*** Read in heating coil inputs
	        xincoil(5) = TWIHTG
            xincoil(6) = WWPRIMHTG
            xincoil(7) = PWOHTG 
	        xincoil(8) = VSTEMHTG
!*** Read in heating coil parameters
            par_vcoil(4) = NROWHTG
	        par_vcoil(5) = NTPRHTG
	        par_vcoil(6) = NCIRHTG
	        par_vcoil(7) = LCOILHTG
	        par_vcoil(17)= FRWHTGCOIL
	        par_vcoil(18)= FRWBYPASHTG 
	        par_vcoil(19)= IVALTYPEHTG
	        par_vcoil(20)= KVHTG
	        par_vcoil(21)= EQPCHARHTG
	        par_vcoil(22)= SVHTG
	        par_vcoil(23)= CLHTG
!*** Call heating coil subroutine  
            CALL TYPE523(xincoil,youtcoil,par_vcoil,saved_vcoil,iostatcoil)
		
!*** Assign output values
            yout(1)  = youtmb(1)
	        yout(2)  = youtmb(2)
            yout(3)  = youtcoil(1)
            yout(4)  = youtcoil(2)
            yout(5)  = youtcoil(3)
            yout(6)  = youtcoil(5)
            yout(7)  = youtcoil(6)
            yout(8)  = youtcoil(7)
            yout(9)  = youtcoil(8)
            yout(10) = youtcoil(9)
            yout(11) = youtcoil(10)
	        yout(12) = youtcoil(11)
	        yout(13) = youtcoil(12)
	        yout(14) = youtcoil(13)
	        yout(15) = youtcoil(14)
	        yout(16) = youtcoil(15)
	                  
        ELSEIF (VSTEMCLG.EQ.0.0.AND.VSTEMHTG.EQ.0.0)THEN
            yout(1)  = youtmb(1)
	        yout(2)  = youtmb(2)
            yout(3)  = youtmb(1)
            yout(4)  = youtmb(1)
            yout(5)  = youtmb(2)
            yout(6)  = -99.9
            yout(7)  = 0.0
            yout(8)  = 0.0
            yout(9)  = -99.9
            yout(10) = 0.0
            yout(11) = 0.0
	        yout(12) = 0.0
	        yout(13) = 0.0
	        yout(14) = -99.9
	        yout(15) = -99.9
	        yout(16) = -99.9
	    ENDIF
 
!*** Assign saved values
!        SAVED(1)=SAVEDFAN(1)
!	  SAVED(2)=SAVEDFAN(2)
!	  SAVED(3)=SAVEDFAN(3)
!	  SAVED(4)=SAVEDCOIL(2)
!	  SAVED(5)=SAVEDCOIL(3)
!	  SAVED(6)=SAVEDCOIL(4)
!	  SAVED(7)=SAVEDCOIL(5)
!	  SAVED(8)=SAVEDCOIL(6)
!	  SAVED(9)=SAVEDCOIL(7)

!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
      return
      end subroutine type310
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a three port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                     (C)
! *  2. G(1)      : Return air humidity ratio                       (kg/kg)
! *  3. W(1)      : Return air mass flow rate                        (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                      (C)
! *  5. G(2)      : Fresh air humidity ratio                        (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                         (kg/s)
! *  7. THERMMODE : FCU thermal mode
! *                 (0=unoccupied,1=cooling,2=heating,3= economizer)    (-)
! *  8. QA        : Fan heat addition rate to fluid stream             (kW)
! *  9. TWICLG    : Cooling coil inlet water temperature                (C)
! * 10. WWPRIMCLG : Cooling coil primary circuit water mass flow rate(kg/s)
! * 11. PWOCLG    : Cooling coil outlet water pressure                (kPa)
! * 12. VSTEMCLG  : Cooling coil valve stem position                    (-)
! * 13. TWIHTG    : Heating coil inlet water temperature                (C)
! * 14. WWPRIMHTG : Heating coil primary circuit water mass flow rate(kg/s)
! * 15. PWOHTG    : Heating coil outlet water pressure                (kPa)
! * 16. VSTEMHTG  : Heating coil valve stem position                    (-)
! * 17. TSDYN     : Effective coil surface temperature                  (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA     : Mixed air dry bulb temperature                        (C)
! *  2. GMA     : Mixed air humidity ratio                          (kg/kg)
! *  3. TS      : Effective coil surface temperature                    (C)
! *  4. TAO     : Fan coil outlet air dry bulb temperature              (C)
! *  5. GO      : Fan coil outlet air humidity ratio                (kg/kg)
! *  6. TWO     : Coil outlet water temperature                         (C)
! *  7. PWI     : Coil inlet water pressure                           (kPa)
! *  8. WW      : Water mass flow rate though coil                   (kg/s)
! *  9. TRET    : Mixed return water temperature                        (C)
! * 10. QA      : Total heat transfer from the coil to the air         (kW)
! * 11. SHR     : Sensible heat ratio of the coil                       (-)
! * 12. EFFECT  : Coil effectiveness                                    (-)
! * 13. BF      : Coil by-pass factor                                   (-)
! * 14. HO      : Fan coil outlet air specific enthalpy             (kJ/kg)
! * 15. RHO     : Fan Coil outlet air relative humidity                 (-)
! * 16. TWBO    : Fan Coil outlet air wet-bulb temperature              (C)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil            (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil            (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil  (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil            (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil            (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil  (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                            (m)
! * 12. HCOIL      : height of finned section                           (m)
! * 13. WCOIL      : width of finned section                            (m)
! * 14. DOTUBE     : tube outside diameter                              (m)
! * 15. THITUBE    : tube wall thickness                                (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 17. SPAFIN     : fin spacing (pitch)                                (m)
! * 18. THIFIN     : fin thickness                                      (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance        (0.001 kg.m)
! * 21. FRWBYPASCLG: cooling coil by-pass water flow resistance(0.001 kg.m)
! * 22. IVALTYPECLG: cooling coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 23. KVCLG      : cooling coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 24. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 25. SVCLG      : cooling coil valve rangability                     (-)
! * 26. CLCLG      : cooling coil valve leakage (fractional flow)       (-)
! * 27. FRWHTGCOIL : heating coil water flow resistance        (0.001 kg.m)
! * 28. FRWBYPASHTG: heating coil by-pass water flow resistance(0.001 kg.m)
! * 29. IVALTYPEHTG: heating coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 30. KVHTG      : heating coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 31. EQPCHARHTG : heating coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 32. SVHTG      : heating coil valve rangability                     (-)
! * 33. CLHTG      : heating coil valve leakage (fractional flow)       (-)
! * 34. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************
!
!        subroutine type311(xin,yout,par_v,saved_v,iostat)
!
!       use modsim_head
!        implicit none
!        integer,parameter                 :: ndtdt=0,ndiffeq=2,&
!                                             ni=17,no=16,np=35,&
!                                             ns=1+ndiffeq*4
!        real,dimension(ni),intent(in)     :: xin                      
!        real,dimension(no),intent(out)    :: yout                     
!        real,dimension(np),intent(in)     :: par_v                    
!        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
!        integer,dimension(35),intent(out) :: iostat
!
!		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
!        real,dimension(nimb)              :: xinmb
!        real,dimension(nomb)              :: youtmb                     
!        real,dimension(npmb)              :: par_vmb
!		real,dimension(nsmb)              :: saved_vmb                  
!        integer,dimension(nomb)           :: iostatmb
!
!	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
!        real,dimension(nifan)             :: xinfan
!        real,dimension(nofan)             :: youtfan                     
!        real,dimension(npfan)             :: par_vfan                    
!        real,dimension(nsfan)             :: saved_vfan                  
!        integer,dimension(nofan)          :: iostatfan
!
!		integer,parameter                 :: nicoil=9,nocoil=15,npcoil=23,nscoil=7
!       real,dimension(nicoil)            :: xincoil
!        real,dimension(nocoil)            :: youtcoil                     
!        real,dimension(npcoil)            :: par_vcoil                    
!        real,dimension(nscoil)            :: saved_vcoil                  
!        integer,dimension(nocoil)         :: iostatcoil
!
!  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
!         real                  :: QA,TWICLG,TRETURN,GRETURN,&
!		                          WWPRIMCLG,PWOCLG,VSTEMCLG,TWIHTG,WWPRIMHTG,&
!								  PWOHTG,VSTEMHTG,TSDYN,TMA,GMA,TS,TAO,GO,TWO,&
!								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
!		
!		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
!		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
!								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
!                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
!	    		
!		integer                :: DYNAMIC,PSYCHRO,IFAULT,THERMMODE
!		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
!		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
!		
!		real                   :: TAI,GI,GSAT 
!	   
!	    real                   :: PATMPA=101325.
!		
!*** Read in mixing box inputs & FCU thermal mode 
!
!        xinmb(1) = xin(1)
!        xinmb(2) = xin(2)
!        xinmb(3) = xin(3)
!        xinmb(4) = xin(4)
!        xinmb(5) = xin(5)
!        xinmb(6) = xin(6)
!		THERMMODE= xin(7)
!		
!*** if FCU is in unoccupied mode damper position is zero and VSTEMCLG/HTG are zero, no need to calculate any thing
!		IF (THERMMODE.EQ.0) THEN
!		    TRETURN  = xin(1)
!            GRETURN  = xin(2)
!            
!			yout(1)  = TRETURN
!	        yout(2)  = GRETURN
!           yout(3)  = TRETURN
!           yout(4)  = TRETURN
!           yout(5)  = GRETURN
!           yout(6)  = -99.9
!           yout(7)  = 0.0
!           yout(8)  = 0.0
!           yout(9)  = -99.9
!           yout(10) = 0.0
!           yout(11) = 0.0
!	        yout(12) = 0.0
!	        yout(13) = 0.0
!	        yout(14) = -99.9
!	        yout(15) = -99.9
!	        yout(16) = -99.9
!       ELSE
!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 
!
!	        CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)
!
!*** Read in fan inputs
!            xinfan(1)= youtmb(3)
!            xinfan(2)= youtmb(1)
!            xinfan(3)= xin(8)
!*** Read in fan parameters
!
!            par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
!	        par_vfan(2)=par_v(34)
!
!*** Call fan subroutine
!            CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
!	  
!*** Read in cooling/heating coil inputs
! 
!            xincoil(1)= youtfan(1)
!            TAI       = youtfan(1)
!	        GI        = youtmb(2)
!*** Limit humidity ratio to saturated value
!            GSAT      = FWPHI(TAI,100.0,PATMPA)
!            GI        = MIN(GI,GSAT)
!            xincoil(2)= GI
!            xincoil(3)= 0
!	        xincoil(4)= youtmb(3)
!	        TWICLG    = xin(9)
!            WWPRIMCLG = xin(10)
!	        PWOCLG    = xin(11)
!            VSTEMCLG  = xin(12)
!	        TWIHTG    = xin(13)
!	        WWPRIMHTG = xin(14)
!            PWOHTG    = xin(15)
!	        VSTEMHTG  = xin(16)
!	        xincoil(9)= xin(17)
!
!*** Read in cooling/heating coil parameters
!            par_vcoil(1) = par_v(1)
!	        par_vcoil(2) = par_v(2)
!	        par_vcoil(3) = par_v(3)
!			NROWCLG      = par_v(4)
!			NTPRCLG      = par_v(5)
!			NCIRCLG      = par_v(6)
!			LCOILCLG     = par_v(7)
!			NROWHTG      = par_v(8)
!			NTPRHTG      = par_v(9)
!			NCIRHTG      = par_v(10)
!			LCOILHTG     = par_v(11)
!			par_vcoil(8) = par_v(12)
!			par_vcoil(9) = par_v(13)
!			par_vcoil(10)= par_v(14)
!			par_vcoil(11)= par_v(15)
!			par_vcoil(12)= par_v(16)
!			par_vcoil(13)= par_v(17)
!			par_vcoil(14)= par_v(18)
!			par_vcoil(15)= par_v(19)
!			par_vcoil(16)= 0
!			FRWCLGCOIL = par_v(20)
!			FRWBYPASCLG= par_v(21)
!			IVALTYPECLG= par_v(22)
!			KVCLG      = par_v(23)
!			EQPCHARCLG = par_v(24)
!			SVCLG      = par_v(25)
!			CLCLG      = par_v(26)
!			FRWHTGCOIL = par_v(27)
!			FRWBYPASHTG= par_v(28)
!			IVALTYPEHTG= par_v(29)
!			KVHTG      = par_v(30)
!			EQPCHARHTG = par_v(31)
!			SVHTG      = par_v(32)
!			CLHTG      = par_v(33)
!        
!	    IF ((VSTEMCLG.NE.0.0).AND.(VSTEMHTG.NE.0.0)) THEN
!	     WRITE(*,*)'TYPE310: Fan coil unit is in both heating and cooling mode!'
!         STOP
!        ENDIF
!        	  
!	        IF (THERMMODE.EQ.1) THEN   
!*** Read in cooling coil inputs
!				xincoil(5) = TWICLG
!				xincoil(6) = WWPRIMCLG
!				xincoil(7) = PWOCLG 
!				xincoil(8) = VSTEMCLG
!*** Read in cooling coil parameters
!				par_vcoil(4) = NROWCLG
!				par_vcoil(5) = NTPRCLG 
!				par_vcoil(6) = NCIRCLG
!				par_vcoil(7) = LCOILCLG
!				par_vcoil(17)= FRWCLGCOIL
!				par_vcoil(18)= FRWBYPASCLG 
!				par_vcoil(19)= IVALTYPECLG
!				par_vcoil(20)= KVCLG
!				par_vcoil(21)= EQPCHARCLG
!				par_vcoil(22)= SVCLG 
!				par_vcoil(23)= CLCLG
!*** Call cooling coil subroutine  
!				CALL TYPE523(xincoil,youtcoil,par_vcoil,saved_vcoil,iostatcoil)
!			
!*** Assign output values
!				yout(1)  = youtmb(1)
!				yout(2)  = youtmb(2)
!				yout(3)  = youtcoil(1)
!				yout(4)  = youtcoil(2)
!				yout(5)  = youtcoil(3)
!				yout(6)  = youtcoil(5)
!				yout(7)  = youtcoil(6)
!				yout(8)  = youtcoil(7)
!				yout(9)  = youtcoil(8)
!				yout(10) = youtcoil(9)
!				yout(11) = youtcoil(10)
!				yout(12) = youtcoil(11)
!				yout(13) = youtcoil(12)
!				yout(14) = youtcoil(13)
!				yout(15) = youtcoil(14)
!				yout(16) = youtcoil(15)
!	                
!	        ELSEIF (THERMMODE.EQ.2) THEN
!*** Read in heating coil inputs
!				xincoil(5) = TWIHTG
!				xincoil(6) = WWPRIMHTG
!				xincoil(7) = PWOHTG 
!				xincoil(8) = VSTEMHTG
!*** Read in heating coil parameters
!				par_vcoil(4) = NROWHTG
!				par_vcoil(5) = NTPRHTG
!				par_vcoil(6) = NCIRHTG
!				par_vcoil(7) = LCOILHTG
!				par_vcoil(17)= FRWHTGCOIL
!				par_vcoil(18)= FRWBYPASHTG 
!				par_vcoil(19)= IVALTYPEHTG
!				par_vcoil(20)= KVHTG
!				par_vcoil(21)= EQPCHARHTG
!				par_vcoil(22)= SVHTG
!				par_vcoil(23)= CLHTG
!*** Call heating coil subroutine  
!				CALL TYPE523(xincoil,youtcoil,par_vcoil,saved_vcoil,iostatcoil)
!		
!*** Assign output values
!				yout(1)  = youtmb(1)
!				yout(2)  = youtmb(2)
!				yout(3)  = youtcoil(1)
!				yout(4)  = youtcoil(2)
!				yout(5)  = youtcoil(3)
!				yout(6)  = youtcoil(5)
!				yout(7)  = youtcoil(6)
!				yout(8)  = youtcoil(7)
!				yout(9)  = youtcoil(8)
!				yout(10) = youtcoil(9)
!				yout(11) = youtcoil(10)
!				yout(12) = youtcoil(11)
!				yout(13) = youtcoil(12)
!				yout(14) = youtcoil(13)
!				yout(15) = youtcoil(14)
!				yout(16) = youtcoil(15)
!	                  
!           ELSEIF (THERMMODE.EQ.3)THEN
!				yout(1)  = youtmb(1)
!				yout(2)  = youtmb(2)
!				yout(3)  = youtmb(1)
!				yout(4)  = youtmb(1)
!				yout(5)  = youtmb(2)
!				yout(6)  = -99.9
!				yout(7)  = 0.0
!				yout(8)  = 0.0
!				yout(9)  = -99.9
!				yout(10) = 0.0
!				yout(11) = 0.0
!				yout(12) = 0.0
!				yout(13) = 0.0
!				yout(14) = -99.9
!				yout(15) = -99.9
!				yout(16) = -99.9
!	        ENDIF
!		ENDIF
! 
!*** Assign saved values
!        SAVED(1)=SAVEDFAN(1)
!	  SAVED(2)=SAVEDFAN(2)
!	  SAVED(3)=SAVEDFAN(3)
!	  SAVED(4)=SAVEDCOIL(2)
!	  SAVED(5)=SAVEDCOIL(3)
!	  SAVED(6)=SAVEDCOIL(4)
!	  SAVED(7)=SAVEDCOIL(5)
!	  SAVED(8)=SAVEDCOIL(6)
!	  SAVED(9)=SAVEDCOIL(7)

!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
!      return
!      end subroutine type311
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a three port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                     (C)
! *  2. G(1)      : Return air humidity ratio                       (kg/kg)
! *  3. W(1)      : Return air mass flow rate                        (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                      (C)
! *  5. G(2)      : Fresh air humidity ratio                        (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                         (kg/s)
! *  7. QA        : Fan heat addition rate to fluid stream             (kW)
! *  8. TWICLG    : Cooling coil inlet water temperature                (C)
! *  9. WWPRIMCLG : Cooling coil primary circuit water mass flow rate(kg/s)
! * 10. PWOCLG    : Cooling coil outlet water pressure                (kPa)
! * 11. VSTEMCLG  : Cooling coil valve stem position                    (-)
! * 12. TSDYNCLG  : Cooling coil effective surface temperature          (C)
! * 13. TWIHTG    : Heating coil inlet water temperature                (C)
! * 14. WWPRIMHTG : Heating coil primary circuit water mass flow rate(kg/s)
! * 15. PWOHTG    : Heating coil outlet water pressure                (kPa)
! * 16. VSTEMHTG  : Heating coil valve stem position                    (-)
! * 17. TSDYNHTG  : Heating coil effective surface temperature          (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA      : Mixed air dry bulb temperature                        (C)
! *  2. GMA      : Mixed air humidity ratio                          (kg/kg)
! *  3. TAO      : Fan coil outlet air dry bulb temperature              (C)
! *  4. GO       : Fan coil outlet air humidity ratio                (kg/kg)
! *  5. TSCLG    : Cooling coil effective surface temperature            (C)
! *  6. TWOCLG   : Cooling coil outlet water temperature                 (C)
! *  7. PWICLG   : Cooling coil inlet water pressure                   (kPa)
! *  8. WWCLG    : Water mass flow rate though cooling coil           (kg/s)
! *  9. TRETCLG  : Mixed return cooling coil water temperature           (C)
! * 10. QACLG    : Total heat transfer from the cooling coil to the air (kW)
! * 11. SHRCLG   : Sensible heat ratio of the cooling coil               (-)
! * 12. EFFECTCLG: Cooling coil effectiveness                            (-)
! * 13. BFCLG    : Cooling coil by-pass factor                           (-)
! * 14. TSHTG    : Heating coil effective surface temperature            (C)
! * 15. TWOHTG   : Heating coil outlet water temperature                 (C)
! * 16. PWIHTG   : Heating coil inlet water pressure                   (kPa)
! * 17. WWHTG    : Water mass flow rate though heating coil           (kg/s)
! * 18. TRETHTG  : Mixed return heating coil water temperature           (C)
! * 19. QAHTG    : Total heat transfer from the heating coil to the air (kW)
! * 20. SHRHTG   : Sensible heat ratio of the heating coil               (-)
! * 21. EFFECTHTG: Heating coil effectiveness                            (-)
! * 22. BFHTG    : Heating coil by-pass factor                           (-)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil            (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil            (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil  (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil            (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil            (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil  (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                            (m)
! * 12. HCOIL      : height of finned section                           (m)
! * 13. WCOIL      : width of finned section                            (m)
! * 14. DOTUBE     : tube outside diameter                              (m)
! * 15. THITUBE    : tube wall thickness                                (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 17. SPAFIN     : fin spacing (pitch)                                (m)
! * 18. THIFIN     : fin thickness                                      (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance        (0.001 kg.m)
! * 21. FRWBYPASCLG: cooling coil by-pass water flow resistance(0.001 kg.m)
! * 22. IVALTYPECLG: cooling coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 23. KVCLG      : cooling coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 24. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 25. SVCLG      : cooling coil valve rangability                     (-)
! * 26. CLCLG      : cooling coil valve leakage (fractional flow)       (-)
! * 27. FRWHTGCOIL : heating coil water flow resistance        (0.001 kg.m)
! * 28. FRWBYPASHTG: heating coil by-pass water flow resistance(0.001 kg.m)
! * 29. IVALTYPEHTG: heating coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 30. KVHTG      : heating coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 31. EQPCHARHTG : heating coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 32. SVHTG      : heating coil valve rangability                     (-)
! * 33. CLHTG      : heating coil valve leakage (fractional flow)       (-)
! * 34. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type312(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=3,&
                                             ni=17,no=22,np=35,&
                                             ns=ndiffeq*3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

		integer,parameter                 :: niclgcoil=9,noclgcoil=15,npclgcoil=23,nsclgcoil=7
        real,dimension(niclgcoil)         :: xinclgcoil
        real,dimension(noclgcoil)         :: youtclgcoil                     
        real,dimension(npclgcoil)         :: par_vclgcoil                    
        real,dimension(nsclgcoil)         :: saved_vclgcoil                  
        integer,dimension(noclgcoil)      :: iostatclgcoil
		
		integer,parameter                 :: nihtgcoil=9,nohtgcoil=15,nphtgcoil=23,nshtgcoil=7
        real,dimension(nihtgcoil)         :: xinhtgcoil
        real,dimension(nohtgcoil)         :: youthtgcoil                     
        real,dimension(nphtgcoil)         :: par_vhtgcoil                    
        real,dimension(nshtgcoil)         :: saved_vhtgcoil                  
        integer,dimension(nohtgcoil)      :: iostathtgcoil

		  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
         real                  :: QA,TSDYNCLG,TSDYNHTG,TWICLG,&
		                          WWPRIMCLG,PWOCLG,VSTEMCLG,TWIHTG,WWPRIMHTG,&
								  PWOHTG,VSTEMHTG,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAICLG,GICLG,TAIHTG,GIHTG,GSAT 
	   
	    real                   :: PATMPA=101325.
		
!*** Read in inputs 
        xinmb(1) = xin(1)
        xinmb(2) = xin(2)
        xinmb(3) = xin(3)
        xinmb(4) = xin(4)
        xinmb(5) = xin(5)
        xinmb(6) = xin(6)
		QA       = xin(7)  
        TWICLG   = xin(8)
        WWPRIMCLG= xin(9) 
        PWOCLG   = xin(10)    
        VSTEMCLG = xin(11)
!*** Fault flag on*********************************
!		IF (VLVcoolcoil.EQ.1) THEN
!*** Cooling valve is stuck
!			VSTEMCLG = VVLVcoolcoil
!		ENDIF
!*** Fault flag off*********************************
        TSDYNCLG = xin(12)  
        TWIHTG   = xin(13)
        WWPRIMHTG= xin(14)
        PWOHTG   = xin(15)
        VSTEMHTG = xin(16)
!*** Fault flag on*********************************
!		IF (VLVheatcoil.EQ.1) THEN
!*** Heating valve is stuck
!			VSTEMHTG = VVLVheatcoil
!		ENDIF
!*** Fault flag off*********************************
		TSDYNHTG = xin(17)

!*** Read in cooling coil parameters
		par_vclgcoil(1) = par_v(1)
		par_vclgcoil(2) = par_v(2)
	    par_vclgcoil(3) = par_v(3)
		par_vclgcoil(4) = par_v(4)
		par_vclgcoil(5) = par_v(5) 
		par_vclgcoil(6) = par_v(6)
		par_vclgcoil(7) = par_v(7)
		par_vclgcoil(8) = par_v(12)
		par_vclgcoil(9) = par_v(13)
		par_vclgcoil(10)= par_v(14)
		par_vclgcoil(11)= par_v(15)
		par_vclgcoil(12)= par_v(16)
		par_vclgcoil(13)= par_v(17)
		par_vclgcoil(14)= par_v(18)
		par_vclgcoil(15)= par_v(19)
		par_vclgcoil(16)= 0.0
		par_vclgcoil(17)= par_v(20)
		par_vclgcoil(18)= par_v(21) 
		par_vclgcoil(19)= par_v(22)
		par_vclgcoil(20)= par_v(23)
		par_vclgcoil(21)= par_v(24)
		par_vclgcoil(22)= par_v(25)
		par_vclgcoil(23)= par_v(26)

!*** Read in heating coil parameters
		par_vhtgcoil(1) = par_v(1)
		par_vhtgcoil(2) = par_v(2)
	    par_vhtgcoil(3) = par_v(3)
        par_vhtgcoil(4) = par_v(8)
		par_vhtgcoil(5) = par_v(9)
		par_vhtgcoil(6) = par_v(10)
		par_vhtgcoil(7) = par_v(11)
		par_vhtgcoil(8) = par_v(12)
		par_vhtgcoil(9) = par_v(13)
		par_vhtgcoil(10)= par_v(14)
		par_vhtgcoil(11)= par_v(15)
		par_vhtgcoil(12)= par_v(16)
		par_vhtgcoil(13)= par_v(17)
		par_vhtgcoil(14)= par_v(18)
		par_vhtgcoil(15)= par_v(19)
		par_vhtgcoil(16)= 0.0
		par_vhtgcoil(17)= par_v(27)
		par_vhtgcoil(18)= par_v(28) 
		par_vhtgcoil(19)= par_v(29)
		par_vhtgcoil(20)= par_v(30)
		par_vhtgcoil(21)= par_v(31)
		par_vhtgcoil(22)= par_v(32)
		par_vhtgcoil(23)= par_v(33)

!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
        
		xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= QA

!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
	    par_vfan(2)=par_v(34)

!*** Call fan subroutine
        
		CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)

!*** Read in cooling coil inputs
		xinclgcoil(1)= youtfan(1)
		TAICLG       = youtfan(1)
		GICLG        = youtmb(2)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAICLG,100.0,PATMPA)
		GICLG        = MIN(GICLG,GSAT)
		xinclgcoil(2)= GICLG
		xinclgcoil(3)= 0.0
		xinclgcoil(4)= youtmb(3)
        xinclgcoil(5)= TWICLG
		xinclgcoil(6)= WWPRIMCLG
		xinclgcoil(7)= PWOCLG 
		xinclgcoil(8)= VSTEMCLG
		xinclgcoil(9)= TSDYNCLG

!*** Call cooling coil subroutine  
        CALL TYPE523(xinclgcoil,youtclgcoil,par_vclgcoil,saved_vclgcoil,iostatclgcoil)

!*** Read in heating coil inputs
		xinhtgcoil(1)= youtclgcoil(2)
		TAIHTG       = youtclgcoil(2)
		GIHTG        = youtclgcoil(3)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAIHTG,100.0,PATMPA)
		GIHTG        = MIN(GIHTG,GSAT)
		xinhtgcoil(2)= GIHTG
		xinhtgcoil(3)= 0.0
		xinhtgcoil(4)= youtmb(3)
		xinhtgcoil(5)= TWIHTG
        xinhtgcoil(6)= WWPRIMHTG
        xinhtgcoil(7)= PWOHTG 
	    xinhtgcoil(8)= VSTEMHTG
		xinhtgcoil(9)= TSDYNHTG

!*** Call heating coil subroutine  
		CALL TYPE523(xinhtgcoil,youthtgcoil,par_vhtgcoil,saved_vhtgcoil,iostathtgcoil)
	  

!*** Assign output values
		yout(1)  = youtmb(1)
		yout(2)  = youtmb(2)
		yout(3)  = youthtgcoil(2)
		yout(4)  = youthtgcoil(3)

		yout(5)  = youtclgcoil(1)
		yout(6)  = youtclgcoil(5)
		yout(7)  = youtclgcoil(6)
		yout(8)  = youtclgcoil(7)
		yout(9)  = youtclgcoil(8)
		yout(10) = youtclgcoil(9)
		yout(11) = youtclgcoil(10)
		yout(12) = youtclgcoil(11)
		yout(13) = youtclgcoil(12)

		yout(14) = youthtgcoil(1)
		yout(15) = youthtgcoil(5)
		yout(16) = youthtgcoil(6)
		yout(17) = youthtgcoil(7)
		yout(18) = youthtgcoil(8)
		yout(19) = youthtgcoil(9)
		yout(20) = youthtgcoil(10)
		yout(21) = youthtgcoil(11)
		yout(22) = youthtgcoil(12)

!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
      return
      end subroutine type312
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a two port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                             (C)
! *  2. G(1)      : Return air humidity ratio                               (kg/kg)
! *  3. W(1)      : Return air mass flow rate                                (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                              (C)
! *  5. G(2)      : Fresh air humidity ratio                                (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                                 (kg/s)
! *  7. QA        : Fan heat addition rate to fluid stream                     (kW)
! *  8. TWICLG    : Cooling coil inlet water temperature                        (C)
! *  9. PWICLG    : Cooling coil inlet water pressure                         (kPa)
! * 10. PWOCLG    : Cooling coil outlet water pressure                        (kPa)
! * 11. VSTEMCLG  : Cooling coil valve stem position                            (-)
! * 12. TSDYNCLG  : Cooling coil effective surface temperature                  (C)
! * 13. TWIHTG    : Heating coil inlet water temperature                        (C)
! * 14. PWIHTG    : Heating coil inlet water pressure                         (kPa)
! * 15. PWOHTG    : Heating coil outlet water pressure                        (kPa)
! * 16. VSTEMHTG  : Heating coil valve stem position                            (-)
! * 17. TSDYNHTG  : Heating coil effective surface temperature                  (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA      : Mixed air dry bulb temperature                               (C)
! *  2. GMA      : Mixed air humidity ratio                                 (kg/kg)
! *  3. TAO      : Fan coil outlet air dry bulb temperature                     (C)
! *  4. GO       : Fan coil outlet air humidity ratio                       (kg/kg)
! *  5. TSCLG    : Cooling coil effective surface temperature                   (C)
! *  6. TWOCLG   : Cooling coil outlet water temperature                        (C)
! *  7. WWCLG    : Water mass flow rate though cooling coil                  (kg/s)
! *  8. QACLG    : Total heat transfer from the cooling coil to the air        (kW)
! *  9. SHRCLG   : Sensible heat ratio of the cooling coil                      (-)
! * 10. EFFECTCLG: Cooling coil effectiveness                                   (-)
! * 11. BFCLG    : Cooling coil by-pass factor                                  (-)
! * 12. TSHTG    : Heating coil effective surface temperature                   (C)
! * 13. TWOHTG   : Heating coil outlet water temperature                        (C)
! * 14. WWHTG    : Water mass flow rate though heating coil                  (kg/s)
! * 15. QAHTG    : Total heat transfer from the heating coil to the air        (kW)
! * 16. SHRHTG   : Sensible heat ratio of the heating coil                      (-)
! * 17. EFFECTHTG: Heating coil effectiveness                                   (-)
! * 18. BFHTG    : Heating coil by-pass factor                                  (-)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                              (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)           (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs        (-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil                    (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil                    (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil          (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                                    (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil                    (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil                    (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil          (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                                    (m)
! * 12. HCOIL      : height of finned section                                   (m)
! * 13. WCOIL      : width of finned section                                    (m)
! * 14. DOTUBE     : tube outside diameter                                      (m)
! * 15. THITUBE    : tube wall thickness                                        (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)                     (-)
! * 17. SPAFIN     : fin spacing (pitch)                                        (m)
! * 18. THIFIN     : fin thickness                                              (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                              (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance                (0.001 kg.m)
! * 21. KVCLG      : cooling coil valve capacity (Kv)               (m3/hr @ 1 bar)
! * 22. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                                    (-)
! * 23. SVCLG      : cooling coil valve rangability                             (-)
! * 24. CLCLG      : cooling coil valve leakage (fractional flow)               (-)
! *******************************&&&&should be added&&&&********************************
! *   . XLIN1CLG   : cooling valve stem position at which second segment starts (-)
! *   . XLIN2CLG   : cooling valve stem position at which third segment starts  (-)
! *   . FRAC2CLG   : cooling valve flow fraction at third segment start position(-)
! *******************************&&&&should be added&&&&********************************
! * 25. FRWHTGCOIL : heating coil water flow resistance                (0.001 kg.m)
! * 26. KVHTG      : heating coil valve capacity (Kv)               (m3/hr @ 1 bar)
! * 27. EQPCHARHTG : heating coil valve curvature parameter   
! *                  (equal percentage port)                                    (-)
! * 28. SVHTG      : heating coil valve rangability                             (-)
! * 29. CLHTG      : heating coil valve leakage (fractional flow)               (-)
! *******************************&&&&should be added&&&&********************************
! *   . XLIN1HTG   : heating valve stem position at which second segment starts (-)
! *   . XLIN2HTG   : heating valve stem position at which third segment starts  (-)
! *   . FRAC2HTG   : heating valve flow fraction at third segment start position(-)
! *******************************&&&&should be added&&&&********************************
! * 30. TCON       : time constant                                              (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type314(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=3,&
                                             ni=17,no=18,np=30,&
                                             ns=ndiffeq*3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

        integer,parameter                 :: niclgcoil=9,noclgcoil=13,npclgcoil=24,nsclgcoil=7
!		integer,parameter                 :: niclgcoil=9,noclgcoil=13,npclgcoil=21,nsclgcoil=7
        real,dimension(niclgcoil)         :: xinclgcoil
        real,dimension(noclgcoil)         :: youtclgcoil                     
        real,dimension(npclgcoil)         :: par_vclgcoil                    
        real,dimension(nsclgcoil)         :: saved_vclgcoil                  
        integer,dimension(noclgcoil)      :: iostatclgcoil
		integer,dimension(2)              :: coolcoilfault

        integer,parameter                 :: nihtgcoil=9,nohtgcoil=13,nphtgcoil=24,nshtgcoil=7
!		integer,parameter                 :: nihtgcoil=9,nohtgcoil=13,nphtgcoil=21,nshtgcoil=7
        real,dimension(nihtgcoil)         :: xinhtgcoil
        real,dimension(nohtgcoil)         :: youthtgcoil                     
        real,dimension(nphtgcoil)         :: par_vhtgcoil                    
        real,dimension(nshtgcoil)         :: saved_vhtgcoil                  
        integer,dimension(nohtgcoil)      :: iostathtgcoil
        integer,dimension(2)              :: heatcoilfault
		  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
        real                   :: QA,TSDYNCLG,TSDYNHTG,TWICLG,&
		                          PWICLG,PWOCLG,VSTEMCLG,TWIHTG,PWIHTG,&
								  PWOHTG,VSTEMHTG,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
								  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
        
! *******************************&&&&should change&&&&&&********************************
!        real                   :: XLIN1CLG,XLIN2CLG,FRAC2CLG,XLIN1HTG,XLIN2HTG,FRAC2HTG

! ***********For now (temporary)********************************************************
		real                   :: XLIN1CLG=0.1,XLIN2CLG=0.8,FRAC2CLG=0.983382
		real                   :: XLIN1HTG=0.1,XLIN2HTG=0.7,FRAC2HTG=0.97033
! *******************************&&&&should change&&&&&&********************************
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAICLG,GICLG,TAIHTG,GIHTG,GSAT 
	   
	    real                   :: PATMPA=101325.
		
!*** Read in inputs 
        xinmb(1) = xin(1)
        xinmb(2) = xin(2)
        xinmb(3) = xin(3)
        xinmb(4) = xin(4)
        xinmb(5) = xin(5)
        xinmb(6) = xin(6)
		QA       = xin(7)  
        TWICLG   = xin(8)
        PWICLG   = xin(9) 
        PWOCLG   = xin(10)    
        VSTEMCLG = xin(11)
        TSDYNCLG = xin(12)  
        TWIHTG   = xin(13)
        PWIHTG   = xin(14)
        PWOHTG   = xin(15)
        VSTEMHTG = xin(16)
		TSDYNHTG = xin(17)

!*** Read in cooling coil parameters
		par_vclgcoil(1) = par_v(1)
		par_vclgcoil(2) = par_v(2)
	    par_vclgcoil(3) = par_v(3)
		par_vclgcoil(4) = par_v(4)
		par_vclgcoil(5) = par_v(5) 
		par_vclgcoil(6) = par_v(6)
		par_vclgcoil(7) = par_v(7)
		par_vclgcoil(8) = par_v(12)
		par_vclgcoil(9) = par_v(13)
		par_vclgcoil(10)= par_v(14)
		par_vclgcoil(11)= par_v(15)
		par_vclgcoil(12)= par_v(16)
		par_vclgcoil(13)= par_v(17)
		par_vclgcoil(14)= par_v(18)
		par_vclgcoil(15)= par_v(19)
		par_vclgcoil(16)= 0.0
		par_vclgcoil(17)= par_v(20)
		par_vclgcoil(18)= par_v(21) 
		par_vclgcoil(19)= par_v(22)
		par_vclgcoil(20)= par_v(23)
		par_vclgcoil(21)= par_v(24)
! *******************************&&&&should change&&&&&&********************************
!		par_vclgcoil(22)= par_v(25)
!		par_vclgcoil(23)= par_v(26)
!		par_vclgcoil(24)= par_v(27)
! ***********For now (temporary)********************************************************
        par_vclgcoil(22)= XLIN1CLG
		par_vclgcoil(23)= XLIN2CLG
		par_vclgcoil(24)= FRAC2CLG
! *******************************&&&&should change&&&&&&********************************
		        
!*** Read in heating coil parameters
		par_vhtgcoil(1) = par_v(1)
		par_vhtgcoil(2) = par_v(2)
	    par_vhtgcoil(3) = par_v(3)
        par_vhtgcoil(4) = par_v(8)
!        par_vhtgcoil(4) = 0.0
		par_vhtgcoil(5) = par_v(9)
		par_vhtgcoil(6) = par_v(10)
		par_vhtgcoil(7) = par_v(11)
		par_vhtgcoil(8) = par_v(12)
		par_vhtgcoil(9) = par_v(13)
		par_vhtgcoil(10)= par_v(14)
		par_vhtgcoil(11)= par_v(15)
		par_vhtgcoil(12)= par_v(16)
		par_vhtgcoil(13)= par_v(17)
		par_vhtgcoil(14)= par_v(18)
		par_vhtgcoil(15)= par_v(19)
		par_vhtgcoil(16)= 0.0
! *******************************&&&&should change&&&&&&********************************
!       par_vhtgcoil(17)= par_v(28)
!		par_vhtgcoil(18)= par_v(29) 
!		par_vhtgcoil(19)= par_v(30)
!		par_vhtgcoil(20)= par_v(31)
!		par_vhtgcoil(21)= par_v(32)
!		par_vhtgcoil(22)= par_v(33)
!		par_vhtgcoil(23)= par_v(34)
!		par_vhtgcoil(24)= par_v(35)
! ***********For now (temporary)********************************************************
        par_vhtgcoil(22)= XLIN1HTG
		par_vhtgcoil(23)= XLIN2HTG
		par_vhtgcoil(24)= FRAC2HTG
! *******************************&&&&should change&&&&&&********************************
		par_vhtgcoil(17)= par_v(25)
		par_vhtgcoil(18)= par_v(26) 
		par_vhtgcoil(19)= par_v(27)
		par_vhtgcoil(20)= par_v(28)
		par_vhtgcoil(21)= par_v(29)

!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
        
		xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= QA

!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
! *******************************&&&&should change&&&&&&********************************
!       par_vfan(2)=par_v(36)
! *******************************&&&&should change&&&&&&********************************
	    par_vfan(2)=par_v(30)

!*** Call fan subroutine
        
		CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)

!*** Read in cooling coil inputs
		xinclgcoil(1)= youtfan(1)
		TAICLG       = youtfan(1)
		GICLG        = youtmb(2)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAICLG,100.0,PATMPA)
		GICLG        = MIN(GICLG,GSAT)
		xinclgcoil(2)= GICLG
		xinclgcoil(3)= 0.0
		xinclgcoil(4)= youtmb(3)
        xinclgcoil(5)= TWICLG
		xinclgcoil(6)= PWICLG
		xinclgcoil(7)= PWOCLG 
		xinclgcoil(8)= VSTEMCLG
		xinclgcoil(9)= TSDYNCLG
!C*** Faults*****************************************************		
		coolcoilfault(1)=coilfault(1)
        coolcoilfault(2)=coilfault(3)
!C***************************************************************
!*** Call cooling coil subroutine  
        CALL TYPE520(xinclgcoil,youtclgcoil,par_vclgcoil,saved_vclgcoil,iostatclgcoil,coolcoilfault,vcoilfault)

!*** Read in heating coil inputs
		xinhtgcoil(1)= youtclgcoil(2)
		TAIHTG       = youtclgcoil(2)
		GIHTG        = youtclgcoil(3)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAIHTG,100.0,PATMPA)
		GIHTG        = MIN(GIHTG,GSAT)
		xinhtgcoil(2)= GIHTG
		xinhtgcoil(3)= 0.0
		xinhtgcoil(4)= youtmb(3)
		xinhtgcoil(5)= TWIHTG
        xinhtgcoil(6)= PWIHTG
        xinhtgcoil(7)= PWOHTG 
	    xinhtgcoil(8)= VSTEMHTG
		xinhtgcoil(9)= TSDYNHTG
!C*** Faults*****************************************************		
		heatcoilfault(1)=coilfault(2)
        heatcoilfault(2)=coilfault(4)
!C***************************************************************

!*** Call heating coil subroutine  
		CALL TYPE520(xinhtgcoil,youthtgcoil,par_vhtgcoil,saved_vhtgcoil,iostathtgcoil,heatcoilfault,vcoilfault)
	  
!*** Assign output values
		yout(1)  = youtmb(1)
		yout(2)  = youtmb(2)
		yout(3)  = youthtgcoil(2)
		yout(4)  = youthtgcoil(3)

		yout(5)  = youtclgcoil(1)
		yout(6)  = youtclgcoil(5)
		yout(7)  = youtclgcoil(6)
		yout(8)  = youtclgcoil(7)
		yout(9)  = youtclgcoil(8)
		yout(10) = youtclgcoil(9)
		yout(11) = youtclgcoil(10)
		
		yout(12) = youthtgcoil(1)
		yout(13) = youthtgcoil(5)
		yout(14) = youthtgcoil(6)
		yout(15) = youthtgcoil(7)
		yout(16) = youthtgcoil(8)
		yout(17) = youthtgcoil(9)
		yout(18) = youthtgcoil(10)

      return
      end subroutine type314
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Series fan powered VAV box dynamic or steady state thermal
! *              calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              fan powered VAV box. Water conditions for reheat coil 
! *              with a two port valve are also calculated.
! *              Calculates inlet pressure and reheat coil water flow             
! *              from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. tplenair  : Plenum air dry bulb temperature                     (C)
! *  2. gplenair  : Plenum air humidity ratio                       (kg/kg)
! *  3. wplenair  : Plenum air mass flow rate                        (kg/s)
! *  4. TWI       : Reheat coil inlet water temperature                 (C)
! *  5. PWI       : Reheat coil inlet water pressure                  (kPa)
! *  6. PWO       : Reheat coil outlet water pressure                 (kPa)
! *  7. VSTEM     : Reheat coil valve stem position                     (-)
! *  8. TSDYN     : Reheat coil effective surface temperature           (C)
! *  9. tprimair  : primary air dry bulb temperature                    (C)
! * 10. gprimair  : primary air humidity ratio                      (kg/kg)
! * 11. wprimair  : primary air mass flow rate                       (kg/s)
! * 12. qa        : added heat to the fluid stream by fan              (Kw)
! * 
! * OUTPUTS
! * =======
! *  1. TS      : Reheat coil effective surface temperature            (C)
! *  2. TAO     : Fan powered unit outlet air dry bulb temperature     (C)
! *  3. GO      : Fan powered unit outlet air humidity ratio       (kg/kg)
! *  4. TWO     : Reheat coil outlet water temperature                (C)
! *  5. WW      : Water mass flow rate though reheat coil           (kg/s)
! *  6. QA      : Total heat transfer from the reheat coil to the air (kW)
! *  7. SHR     : Sensible heat ratio of the reheat coil               (-)
! *  8. EFFECT  : Reheat coil effectiveness                            (-)
! *  9. BF      : Reheat coil by-pass factor                           (-)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROW       : number of rows of tubes in coil                    (-)
! *  5. NTPR       : number of tubes per row in coil                    (-)
! *  6. NCIR       : number of parallel water circuits in coil          (-)
! *  7. LCOIL      : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. HCOIL      : height of finned section                           (m)
! *  9. WCOIL      : width of finned section                            (m)
! * 10. DOTUBE     : tube outside diameter                              (m)
! * 11. THITUBE    : tube wall thickness                                (m)
! * 12. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. SPAFIN     : fin spacing (pitch)                                (m)
! * 14. THIFIN     : fin thickness                                      (m)
! * 15. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. FRW        : coil water flow resistance                (0.001 kg.m)
! * 17. KV         : coil valve capacity (Kv)               (m3/hr @ 1 bar)
! * 18. EQPCHAR    : coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 19. SV         : coil valve rangability                             (-)
! * 20. CL         : coil valve leakage (fractional flow)               (-)
! * 21. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                Aug 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 522
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type544(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=3,&
                                             ni=12,no=9,np=21,&
                                             ns=ndiffeq*3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

		integer,parameter                 :: nicoil=9,nocoil=13,npcoil=21,nscoil=7
        real,dimension(nicoil)            :: xincoil
        real,dimension(nocoil)            :: youtcoil                     
        real,dimension(npcoil)            :: par_vcoil                    
        real,dimension(nscoil)            :: saved_vcoil                  
        integer,dimension(nocoil)         :: iostatcoil
		
				  

        real                   :: QA,TSDYNCLG,TSDYNHTG,TWICLG,&
		                          PWICLG,PWOCLG,VSTEMCLG,TWIHTG,PWIHTG,&
								  PWOHTG,VSTEMHTG,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAICLG,GICLG,TAIHTG,GIHTG,GSAT 
	   
	    real                   :: PATMPA=101325.
		
	
!*** Read in coil inputs 
        xincoil(1) = xin(1)
        xincoil(2) = xin(2)
        xincoil(3) = 0        !outlet air pressure
        xincoil(4) = xin(3)
        xincoil(5) = xin(4)
        xincoil(6) = xin(5)
		xincoil(7) = xin(6)
        xincoil(8) = xin(7)
        xincoil(9) = xin(8)

!*** Read in coil parameters
		par_vcoil(1) = par_v(1)
		par_vcoil(2) = par_v(2)
	    par_vcoil(3) = par_v(3)
		par_vcoil(4) = par_v(4)
		par_vcoil(5) = par_v(5) 
		par_vcoil(6) = par_v(6)
		par_vcoil(7) = par_v(7)
		par_vcoil(8) = par_v(8)
		par_vcoil(9) = par_v(9)
		par_vcoil(10)= par_v(10)
		par_vcoil(11)= par_v(11)
		par_vcoil(12)= par_v(12)
		par_vcoil(13)= par_v(13)
		par_vcoil(14)= par_v(14)
		par_vcoil(15)= par_v(15)
		par_vcoil(16)= 0.0     ! air side fluid resistance
		par_vcoil(17)= par_v(16)
		par_vcoil(18)= par_v(17) 
		par_vcoil(19)= par_v(18)
		par_vcoil(20)= par_v(19)
		par_vcoil(21)= par_v(20)

!*** Call coil subroutine
  
        CALL TYPE522(xincoil,youtcoil,par_vcoil,saved_vcoil,iostatcoil)

!*** Read in mixing subroutine inputs

        xinmb(1) = youtcoil(2) 
		xinmb(2) = youtcoil(3) 
        xinmb(3) = xin(3)
		xinmb(4) = xin(9)
		xinmb(5) = xin(10)
		xinmb(6) = xin(11)
		
!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
   
		xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= xin(12)

!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
	    par_vfan(2)=par_v(21)

!*** Call fan subroutine
        
		CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)

!*** Assign output values
		yout(1)  = youtcoil(1)
		yout(2)  = youtfan(1)
		yout(3)  = youtmb(2)
		yout(4)  = youtcoil(5)
		yout(5)  = youtcoil(6)
		yout(6)  = youtcoil(7)
		yout(7)  = youtcoil(8)
		yout(8)  = youtcoil(9)
		yout(9)  = youtcoil(10)
				
!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
      return
      end subroutine type544
!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a two port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                     (C)
! *  2. G(1)      : Return air humidity ratio                       (kg/kg)
! *  3. W(1)      : Return air mass flow rate                        (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                      (C)
! *  5. G(2)      : Fresh air humidity ratio                        (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                         (kg/s)
! *  7. QA        : Fan heat addition rate to fluid stream             (kW)
! *  8. TWICLG    : Cooling coil inlet water temperature                (C)
! *  9. PWICLG    : Cooling coil inlet water pressure                 (kPa)
! * 10. PWOCLG    : Cooling coil outlet water pressure                (kPa)
! * 11. VSTEMCLG  : Cooling coil valve stem position                    (-)
! * 12. TSDYNCLG  : Cooling coil effective surface temperature          (C)
! * 13. TWIHTG    : Heating coil inlet water temperature                (C)
! * 14. PWIHTG    : Heating coil inlet water pressure                 (kPa)
! * 15. PWOHTG    : Heating coil outlet water pressure                (kPa)
! * 16. VSTEMHTG  : Heating coil valve stem position                    (-)
! * 17. TSDYNHTG  : Heating coil effective surface temperature          (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA      : Mixed air dry bulb temperature                        (C)
! *  2. GMA      : Mixed air humidity ratio                          (kg/kg)
! *  3. TAO      : Fan coil outlet air dry bulb temperature              (C)
! *  4. GO       : Fan coil outlet air humidity ratio                (kg/kg)
! *  5. TSCLG    : Cooling coil effective surface temperature            (C)
! *  6. TWOCLG   : Cooling coil outlet water temperature                 (C)
! *  7. WWCLG    : Water mass flow rate though cooling coil           (kg/s)
! *  8. QACLG    : Total heat transfer from the cooling coil to the air (kW)
! *  9. SHRCLG   : Sensible heat ratio of the cooling coil               (-)
! * 10. EFFECTCLG: Cooling coil effectiveness                            (-)
! * 11. BFCLG    : Cooling coil by-pass factor                           (-)
! * 12. TSHTG    : Heating coil effective surface temperature            (C)
! * 13. TWOHTG   : Heating coil outlet water temperature                 (C)
! * 14. WWHTG    : Water mass flow rate though heating coil           (kg/s)
! * 15. QAHTG    : Total heat transfer from the heating coil to the air (kW)
! * 16. SHRHTG   : Sensible heat ratio of the heating coil               (-)
! * 17. EFFECTHTG: Heating coil effectiveness                            (-)
! * 18. BFHTG    : Heating coil by-pass factor                           (-)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil            (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil            (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil  (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil            (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil            (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil  (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                            (m)
! * 12. HCOIL      : height of finned section                           (m)
! * 13. WCOIL      : width of finned section                            (m)
! * 14. DOTUBE     : tube outside diameter                              (m)
! * 15. THITUBE    : tube wall thickness                                (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 17. SPAFIN     : fin spacing (pitch)                                (m)
! * 18. THIFIN     : fin thickness                                      (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance        (0.001 kg.m)
! * 21. KVCLG      : cooling coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 22. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 23. SVCLG      : cooling coil valve rangability                     (-)
! * 24. CLCLG      : cooling coil valve leakage (fractional flow)       (-)
! * 25. FRWHTGCOIL : heating coil water flow resistance        (0.001 kg.m)
! * 26. KVHTG      : heating coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 27. EQPCHARHTG : heating coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 28. SVHTG      : heating coil valve rangability                     (-)
! * 29. CLHTG      : heating coil valve leakage (fractional flow)       (-)
! * 30. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type315(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=3,&
                                             ni=17,no=18,np=30,&
                                             ns=ndiffeq*3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

		integer,parameter                 :: niclgcoil=9,noclgcoil=13,npclgcoil=21,nsclgcoil=7
        real,dimension(niclgcoil)         :: xinclgcoil
        real,dimension(noclgcoil)         :: youtclgcoil                     
        real,dimension(npclgcoil)         :: par_vclgcoil                    
        real,dimension(nsclgcoil)         :: saved_vclgcoil                  
        integer,dimension(noclgcoil)      :: iostatclgcoil
		
		integer,parameter                 :: nihtgcoil=9,nohtgcoil=13,nphtgcoil=21,nshtgcoil=7
        real,dimension(nihtgcoil)         :: xinhtgcoil
        real,dimension(nohtgcoil)         :: youthtgcoil                     
        real,dimension(nphtgcoil)         :: par_vhtgcoil                    
        real,dimension(nshtgcoil)         :: saved_vhtgcoil                  
        integer,dimension(nohtgcoil)      :: iostathtgcoil

		  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
         real                  :: QA,TSDYNCLG,TSDYNHTG,TWICLG,&
		                          WWPRIMCLG,PWICLG,PWOCLG,VSTEMCLG,TWIHTG,WWPRIMHTG,&
								  PWIHTG,PWOHTG,VSTEMHTG,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAICLG,GICLG,TAIHTG,GIHTG,GSAT 
	   
	    real                   :: PATMPA=101325.

!*** Read in inputs 
        xinmb(1) = xin(1)
        xinmb(2) = xin(2)
        xinmb(3) = xin(3)
        xinmb(4) = xin(4)
        xinmb(5) = xin(5)
        xinmb(6) = xin(6)
		QA       = xin(7)  
        TWICLG   = xin(8)
        PWICLG   = xin(9) 
        PWOCLG   = xin(10)    
        VSTEMCLG = xin(11)
        TSDYNCLG = xin(12)  
        TWIHTG   = xin(13)
        PWIHTG   = xin(14)
        PWOHTG   = xin(15)
        VSTEMHTG = xin(16)
		TSDYNHTG = xin(17)

!*** Read in cooling coil parameters
		par_vclgcoil(1) = par_v(1)
		par_vclgcoil(2) = par_v(2)
	    par_vclgcoil(3) = par_v(3)
		par_vclgcoil(4) = par_v(4)
		par_vclgcoil(5) = par_v(5) 
		par_vclgcoil(6) = par_v(6)
		par_vclgcoil(7) = par_v(7)
		par_vclgcoil(8) = par_v(12)
		par_vclgcoil(9) = par_v(13)
		par_vclgcoil(10)= par_v(14)
		par_vclgcoil(11)= par_v(15)
		par_vclgcoil(12)= par_v(16)
		par_vclgcoil(13)= par_v(17)
		par_vclgcoil(14)= par_v(18)
		par_vclgcoil(15)= par_v(19)
		par_vclgcoil(16)= 0.0
		par_vclgcoil(17)= par_v(20)
		par_vclgcoil(18)= par_v(21) 
		par_vclgcoil(19)= par_v(22)
		par_vclgcoil(20)= par_v(23)
		par_vclgcoil(21)= par_v(24)
		        
!*** Read in heating coil parameters
		par_vhtgcoil(1) = par_v(1)
		par_vhtgcoil(2) = par_v(2)
	    par_vhtgcoil(3) = par_v(3)
        par_vhtgcoil(4) = par_v(8)
		par_vhtgcoil(5) = par_v(9)
		par_vhtgcoil(6) = par_v(10)
		par_vhtgcoil(7) = par_v(11)
		par_vhtgcoil(8) = par_v(12)
		par_vhtgcoil(9) = par_v(13)
		par_vhtgcoil(10)= par_v(14)
		par_vhtgcoil(11)= par_v(15)
		par_vhtgcoil(12)= par_v(16)
		par_vhtgcoil(13)= par_v(17)
		par_vhtgcoil(14)= par_v(18)
		par_vhtgcoil(15)= par_v(19)
		par_vhtgcoil(16)= 0.0
		par_vhtgcoil(17)= par_v(25)
		par_vhtgcoil(18)= par_v(26) 
		par_vhtgcoil(19)= par_v(27)
		par_vhtgcoil(20)= par_v(28)
		par_vhtgcoil(21)= par_v(29)
				
!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
        
		xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= QA

!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
	    par_vfan(2)=par_v(30)

!*** Call fan subroutine
        
		CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)

!*** Read in cooling coil inputs
		xinclgcoil(1)= youtfan(1)
		TAICLG       = youtfan(1)
		GICLG        = youtmb(2)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAICLG,100.0,PATMPA)
		GICLG        = MIN(GICLG,GSAT)
		xinclgcoil(2)= GICLG
		xinclgcoil(3)= 0.0
		xinclgcoil(4)= youtmb(3)
        xinclgcoil(5)= TWICLG
		xinclgcoil(6)= PWICLG
		xinclgcoil(7)= PWOCLG 
		xinclgcoil(8)= VSTEMCLG
		xinclgcoil(9)= TSDYNCLG

!*** Call cooling coil subroutine  
        CALL TYPE520(xinclgcoil,youtclgcoil,par_vclgcoil,saved_vclgcoil,iostatclgcoil,coilfault,vcoilfault)

!*** Read in heating coil inputs
		xinhtgcoil(1)= youtclgcoil(2)
		TAIHTG       = youtclgcoil(2)
		GIHTG        = youtclgcoil(3)

!*** Limit humidity ratio to saturated value
		GSAT         = FWPHI(TAIHTG,100.0,PATMPA)
		GIHTG        = MIN(GIHTG,GSAT)
		xinhtgcoil(2)= GIHTG
		xinhtgcoil(3)= 0.0
		xinhtgcoil(4)= youtmb(3)
		xinhtgcoil(5)= TWIHTG
        xinhtgcoil(6)= PWIHTG
        xinhtgcoil(7)= PWOHTG 
	    xinhtgcoil(8)= VSTEMHTG
		xinhtgcoil(9)= TSDYNHTG

!*** Call heating coil subroutine  
		CALL TYPE520(xinhtgcoil,youthtgcoil,par_vhtgcoil,saved_vhtgcoil,iostathtgcoil,coilfault,vcoilfault)
	  
!*** Assign output values
		yout(1)  = youtmb(1)
		yout(2)  = youtmb(2)
		yout(3)  = youthtgcoil(2)
		yout(4)  = youthtgcoil(3)

		yout(5)  = youtclgcoil(1)
		yout(6)  = youtclgcoil(5)
		yout(7)  = youtclgcoil(6)
		yout(8)  = youtclgcoil(7)
		yout(9)  = youtclgcoil(8)
		yout(10) = youtclgcoil(9)
		yout(11) = youtclgcoil(10)
		
		yout(12) = youthtgcoil(1)
		yout(13) = youthtgcoil(5)
		yout(14) = youthtgcoil(6)
		yout(15) = youthtgcoil(7)
		yout(16) = youthtgcoil(8)
		yout(17) = youthtgcoil(9)
		yout(18) = youthtgcoil(10)
		
!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
      return
      end subroutine type315

!*************************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!*************************************************************************
! * SUBROUTINE:  Fan coil unit dynamic or steady state thermal calculation
! *
! * LANGUAGE:    FORTRAN 77
! *
! * PURPOSE:     Calculates the outlet air and mixed air conditions of a 
! *              four pipe fan coil unit. Water conditions for heating/
! *              cooling coil with a three port valve are also calculated.
! *              Calculates inlet pressure and heating/cooling coil water            
! *              flow from primary circuit flow rate and outlet pressure.
! ************************************************************************
! * INPUTS
! * ======
! *  1. T(1)      : Return air dry bulb temperature                     (C)
! *  2. G(1)      : Return air humidity ratio                       (kg/kg)
! *  3. W(1)      : Return air mass flow rate                        (kg/s)
! *  4. T(2)      : Fresh air dry bulb temperature                      (C)
! *  5. G(2)      : Fresh air humidity ratio                        (kg/kg)
! *  6. W(2)      : Fresh air mass flow rate                         (kg/s)
! *  7. QA        : Fan heat addition rate to fluid stream             (kW)
! *  8. TWICLG    : Cooling coil inlet water temperature                (C)
! *  9. WWPRIMCLG : Cooling coil primary circuit water mass flow rate(kg/s)
! * 10. PWOCLG    : Cooling coil outlet water pressure                (kPa)
! * 11. VSTEMCLG  : Cooling coil valve stem position                    (-)
! * 12. TSDYNCLG  : Cooling coil effective surface temperature          (C)
! * 13. TWIHTG    : Heating coil inlet water temperature                (C)
! * 14. WWPRIMHTG : Heating coil primary circuit water mass flow rate(kg/s)
! * 15. PWOHTG    : Heating coil outlet water pressure                (kPa)
! * 16. VSTEMHTG  : Heating coil valve stem position                    (-)
! * 17. TSDYNHTG  : Heating coil effective surface temperature          (C)
! *
! * OUTPUTS
! * =======
! *  1. TMA      : Mixed air dry bulb temperature                        (C)
! *  2. GMA      : Mixed air humidity ratio                          (kg/kg)
! *  3. TAO      : Fan coil outlet air dry bulb temperature              (C)
! *  4. GO       : Fan coil outlet air humidity ratio                (kg/kg)
! *  5. TSCLG    : Cooling coil effective surface temperature            (C)
! *  6. TWOCLG   : Cooling coil outlet water temperature                 (C)
! *  7. PWICLG   : Cooling coil inlet water pressure                   (kPa)
! *  8. WWCLG    : Water mass flow rate though cooling coil           (kg/s)
! *  9. TRETCLG  : Mixed return cooling coil water temperature           (C)
! * 10. QACLG    : Total heat transfer from the cooling coil to the air (kW)
! * 11. SHRCLG   : Sensible heat ratio of the cooling coil               (-)
! * 12. EFFECTCLG: Cooling coil effectiveness                            (-)
! * 13. BFCLG    : Cooling coil by-pass factor                           (-)
! * 14. TSHTG    : Heating coil effective surface temperature            (C)
! * 15. TWOHTG   : Heating coil outlet water temperature                 (C)
! * 16. PWIHTG   : Heating coil inlet water pressure                   (kPa)
! * 17. WWHTG    : Water mass flow rate though heating coil           (kg/s)
! * 18. TRETHTG  : Mixed return heating coil water temperature           (C)
! * 19. QAHTG    : Total heat transfer from the heating coil to the air (kW)
! * 20. SHRHTG   : Sensible heat ratio of the heating coil               (-)
! * 21. EFFECTHTG: Heating coil effectiveness                            (-)
! * 22. BFHTG    : Heating coil by-pass factor                           (-)
! *
! * PARAMETERS
! * ==========
! *  1. DYNAMIC    : 0 = steady state, 1 = dynamic                      (-)
! *  2. IFAULT     : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. PSYCHRO    : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. NROWCLG    : number of rows of tubes in cooling coil            (-)
! *  5. NTPRCLG    : number of tubes per row in cooling coil            (-)
! *  6. NCIRCLG    : number of parallel water circuits in cooling coil  (-)
! *  7. LCOILCLG   : length of cooling coil finned section in direction 
! *                  of flow                                            (m)
! *  8. NROWHTG    : number of rows of tubes in heating coil            (-)
! *  9. NTPRHTG    : number of tubes per row in heating coil            (-)
! * 10. NCIRHTG    : number of parallel water circuits in heating coil  (-)
! * 11. LCOILHTG   : length of heating coil finned section in direction 
! *                  of flow                                            (m)
! * 12. HCOIL      : height of finned section                           (m)
! * 13. WCOIL      : width of finned section                            (m)
! * 14. DOTUBE     : tube outside diameter                              (m)
! * 15. THITUBE    : tube wall thickness                                (m)
! * 16. MATUBE     : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 17. SPAFIN     : fin spacing (pitch)                                (m)
! * 18. THIFIN     : fin thickness                                      (m)
! * 19. MAFIN      : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 20. FRWCLGCOIL : cooling coil water flow resistance        (0.001 kg.m)
! * 21. FRWBYPASCLG: cooling coil by-pass water flow resistance(0.001 kg.m)
! * 22. IVALTYPECLG: cooling coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 23. KVCLG      : cooling coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 24. EQPCHARCLG : cooling coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 25. SVCLG      : cooling coil valve rangability                     (-)
! * 26. CLCLG      : cooling coil valve leakage (fractional flow)       (-)
! * 27. FRWHTGCOIL : heating coil water flow resistance        (0.001 kg.m)
! * 28. FRWBYPASHTG: heating coil by-pass water flow resistance(0.001 kg.m)
! * 29. IVALTYPEHTG: heating coil valve type: 0=lin/lin, 
! *                  1=eq%(flow)/lin(byp), 2=lin/eq%                    (-)
! * 30. KVHTG      : heating coil valve capacity (Kv)       (m3/hr @ 1 bar)
! * 31. EQPCHARHTG : heating coil valve curvature parameter 
! *                  (equal percentage port)                            (-)
! * 32. SVHTG      : heating coil valve rangability                     (-)
! * 33. CLHTG      : heating coil valve leakage (fractional flow)       (-)
! * 34. TCON       : time constant                                      (s)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. TIME    : time of previous call
! *  2. DT      : solution of fan differential equation from previous call
! *  3. DTP     : solution of fan differential equation from previous step time
! *  4. DT      : solution of coil differential equation from previous call
! *  5. DTP     : solution of coil differential equation from previous step time
! *  6. AA      : A coefficent in dT/dt=A*T+B from previous call
! *  7. AAP     : A coefficent in dT/dt=A*T+B from previous step time
! *  8. BB      : B coefficent in dT/dt=A*T+B from previous call
! *  9. BBP     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University 
!
!   DATE:                May 2012
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  TYPE 366,TYPE 367,TYPE 523
!   FUNCTIONS  CALLED:   FWPHI
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!***********************************************************************

        subroutine type313(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=2,&
                                             ni=17,no=22,np=35,&
                                             ns=1+ndiffeq*4
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
!        integer,dimension(no),intent(out) :: iostat
        integer,dimension(35),intent(out) :: iostat

		integer,parameter                 :: nimb=6,nomb=3,npmb=1,nsmb=0
        real,dimension(nimb)              :: xinmb
        real,dimension(nomb)              :: youtmb                     
        real,dimension(npmb)              :: par_vmb
		real,dimension(nsmb)              :: saved_vmb                  
        integer,dimension(nomb)           :: iostatmb

	    integer,parameter                 :: nifan=3,nofan=1,npfan=2,nsfan=3
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

		integer,parameter                 :: niclgcoil=9,noclgcoil=15,npclgcoil=23,nsclgcoil=7
        real,dimension(niclgcoil)         :: xinclgcoil
        real,dimension(noclgcoil)         :: youtclgcoil                     
        real,dimension(npclgcoil)         :: par_vclgcoil                    
        real,dimension(nsclgcoil)         :: saved_vclgcoil                  
        integer,dimension(noclgcoil)      :: iostatclgcoil
		
		integer,parameter                 :: nihtgcoil=9,nohtgcoil=15,nphtgcoil=23,nshtgcoil=7
        real,dimension(nihtgcoil)         :: xinhtgcoil
        real,dimension(nohtgcoil)         :: youthtgcoil                     
        real,dimension(nphtgcoil)         :: par_vhtgcoil                    
        real,dimension(nshtgcoil)         :: saved_vhtgcoil                  
        integer,dimension(nohtgcoil)      :: iostathtgcoil

		  
!        real                   :: T(1),G(1),W(1),T(2),G(2),W(2),QA,TWICLG,&
         real                  :: QA,TSDYNCLG,TSDYNHTG,TWICLG,&
		                          WWPRIMCLG,PWOCLG,VSTEMCLG,TWIHTG,WWPRIMHTG,&
								  PWOHTG,VSTEMHTG,TMA,GMA,TS,TAO,GO,TWO,&
								  PWI,WW,TRET,SHR,EFFECT,BF,HO,RHO,TWBO
		
		real                   :: LCOILCLG,LCOILHTG,HCOIL,WCOIL,DOTUBE,THITUBE,&
		                          SPAFIN,THIFIN,FRWCLGCOIL,FRWBYPASCLG,KVCLG,&
								  EQPCHARCLG,SVCLG,CLCLG,FRWHTGCOIL,FRWBYPASHTG,&
                                  KVHTG,EQPCHARHTG,SVHTG,CLHTG,TCON,FWPHI
	    		
		integer                :: DYNAMIC,PSYCHRO,IFAULT
		integer                :: NROWCLG,NTPRCLG,NCIRCLG,NROWHTG,NTPRHTG,&
		                          NCIRHTG,MATUBE,MAFIN,IVALTYPECLG,IVALTYPEHTG,I
		
		real                   :: TAICLG,GICLG,TAIHTG,GIHTG,GSAT 
	   
	    real                   :: PATMPA=101325.
		
!*** Read in inputs 
 
        xinmb(1) = xin(1)
        xinmb(2) = xin(2)
        xinmb(3) = xin(3)
        xinmb(4) = xin(4)
        xinmb(5) = xin(5)
        xinmb(6) = xin(6)
		QA       = xin(7)  
        TWICLG   = xin(8)
        WWPRIMCLG= xin(9) 
        PWOCLG   = xin(10)    
        VSTEMCLG = xin(11)
        TSDYNCLG = xin(12)  
        TWIHTG   = xin(13)
        WWPRIMHTG= xin(14)
        PWOHTG   = xin(15)
        VSTEMHTG = xin(16)
		TSDYNHTG = xin(17)

!*** Call mixing box subroutine to calculate mixing air temperature, humidity ratio and mass flow rate 

	    CALL TYPE367(xinmb,youtmb,par_vmb,saved_vmb,iostatmb)

!*** Read in fan inputs
        
		xinfan(1)= youtmb(3)
        xinfan(2)= youtmb(1)
        xinfan(3)= QA

!*** Read in fan parameters

        par_vfan(1)=1     !FLUID MODE IS 1 BECAUSE IT IS AIR
	    par_vfan(2)=par_v(34)

!*** Call fan subroutine
        
		CALL TYPE366(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	  

		IF ((VSTEMCLG.NE.0.0).AND.(VSTEMHTG.NE.0.0)) THEN
		    
!*** Read in cooling coil inputs
				xinclgcoil(1)= youtfan(1)
				TAICLG       = youtfan(1)
			    GICLG        = youtmb(2)

!*** Limit humidity ratio to saturated value
				GSAT         = FWPHI(TAICLG,100.0,PATMPA)
				GICLG        = MIN(GICLG,GSAT)
				xinclgcoil(2)= GICLG
				xinclgcoil(3)= 0.0
				xinclgcoil(4)= youtmb(3)
                xinclgcoil(5)= TWICLG
		        xinclgcoil(6)= WWPRIMCLG
			    xinclgcoil(7)= PWOCLG 
				xinclgcoil(8)= VSTEMCLG
				xinclgcoil(9)= TSDYNCLG
!*** Read in cooling coil parameters
				par_vclgcoil(1) = par_v(1)
				par_vclgcoil(2) = par_v(2)
	            par_vclgcoil(3) = par_v(3)
				par_vclgcoil(4) = par_v(4)
				par_vclgcoil(5) = par_v(5) 
				par_vclgcoil(6) = par_v(6)
				par_vclgcoil(7) = par_v(7)
				par_vclgcoil(8) = par_v(12)
				par_vclgcoil(9) = par_v(13)
				par_vclgcoil(10)= par_v(14)
				par_vclgcoil(11)= par_v(15)
				par_vclgcoil(12)= par_v(16)
				par_vclgcoil(13)= par_v(17)
				par_vclgcoil(14)= par_v(18)
				par_vclgcoil(15)= par_v(19)
				par_vclgcoil(16)= 0.0
				par_vclgcoil(17)= par_v(20)
				par_vclgcoil(18)= par_v(21) 
				par_vclgcoil(19)= par_v(22)
				par_vclgcoil(20)= par_v(23)
				par_vclgcoil(21)= par_v(24)
				par_vclgcoil(22)= par_v(25)
				par_vclgcoil(23)= par_v(26)

!*** Call cooling coil subroutine  
                CALL TYPE523(xinclgcoil,youtclgcoil,par_vclgcoil,saved_vclgcoil,iostatclgcoil)

!*** Read in heating coil inputs
				xinhtgcoil(1)= youtclgcoil(2)
				TAIHTG       = youtclgcoil(2)
			    GIHTG        = youtclgcoil(3)

!*** Limit humidity ratio to saturated value
				GSAT         = FWPHI(TAIHTG,100.0,PATMPA)
				GIHTG        = MIN(GIHTG,GSAT)
				xinhtgcoil(2)= GIHTG
				xinhtgcoil(3)= 0.0
				xinhtgcoil(4)= youtmb(3)
				xinhtgcoil(5)= TWIHTG
                xinhtgcoil(6)= WWPRIMHTG
                xinhtgcoil(7)= PWOHTG 
	            xinhtgcoil(8)= VSTEMHTG
				xinhtgcoil(9)= TSDYNHTG
!*** Read in heating coil parameters
				par_vhtgcoil(1) = par_v(1)
				par_vhtgcoil(2) = par_v(2)
	            par_vhtgcoil(3) = par_v(3)
                par_vhtgcoil(4) = par_v(8)
				par_vhtgcoil(5) = par_v(9)
				par_vhtgcoil(6) = par_v(10)
				par_vhtgcoil(7) = par_v(11)
				par_vhtgcoil(8) = par_v(12)
				par_vhtgcoil(9) = par_v(13)
				par_vhtgcoil(10)= par_v(14)
				par_vhtgcoil(11)= par_v(15)
				par_vhtgcoil(12)= par_v(16)
				par_vhtgcoil(13)= par_v(17)
				par_vhtgcoil(14)= par_v(18)
				par_vhtgcoil(15)= par_v(19)
				par_vhtgcoil(16)= 0.0
				par_vhtgcoil(17)= par_v(27)
				par_vhtgcoil(18)= par_v(28) 
				par_vhtgcoil(19)= par_v(29)
				par_vhtgcoil(20)= par_v(30)
				par_vhtgcoil(21)= par_v(31)
				par_vhtgcoil(22)= par_v(32)
				par_vhtgcoil(23)= par_v(33)
!*** Call heating coil subroutine  
				CALL TYPE523(xinhtgcoil,youthtgcoil,par_vhtgcoil,saved_vhtgcoil,iostathtgcoil)
!*** Assign output values
				yout(1)  = youtmb(1)
				yout(2)  = youtmb(2)
				yout(3)  = youthtgcoil(2)
				yout(4)  = youthtgcoil(3)

				yout(5)  = youtclgcoil(1)
				yout(6)  = youtclgcoil(5)
				yout(7)  = youtclgcoil(6)
				yout(8)  = youtclgcoil(7)
				yout(9)  = youtclgcoil(8)
				yout(10) = youtclgcoil(9)
				yout(11) = youtclgcoil(10)
				yout(12) = youtclgcoil(11)
				yout(13) = youtclgcoil(12)

				yout(14) = youthtgcoil(1)
				yout(15) = youthtgcoil(5)
				yout(16) = youthtgcoil(6)
				yout(17) = youthtgcoil(7)
				yout(18) = youthtgcoil(8)
				yout(19) = youthtgcoil(9)
				yout(20) = youthtgcoil(10)
				yout(21) = youthtgcoil(11)
				yout(22) = youthtgcoil(12)

		
		ELSEIF ((VSTEMCLG.NE.0.0).AND.(VSTEMHTG.EQ.0.0)) THEN
!*** Read in cooling coil inputs
				xinclgcoil(1)= youtfan(1)
				TAICLG       = youtfan(1)
			    GICLG        = youtmb(2)

!*** Limit humidity ratio to saturated value
				GSAT         = FWPHI(TAICLG,100.0,PATMPA)
				GICLG        = MIN(GICLG,GSAT)
				xinclgcoil(2)= GICLG
				xinclgcoil(3)= 0.0
				xinclgcoil(4)= youtmb(3)
                xinclgcoil(5)= TWICLG
		        xinclgcoil(6)= WWPRIMCLG
			    xinclgcoil(7)= PWOCLG 
				xinclgcoil(8)= VSTEMCLG
				xinclgcoil(9)= TSDYNCLG
!*** Read in cooling coil parameters
				par_vclgcoil(1) = par_v(1)
				par_vclgcoil(2) = par_v(2)
	            par_vclgcoil(3) = par_v(3)
				par_vclgcoil(4) = par_v(4)
				par_vclgcoil(5) = par_v(5) 
				par_vclgcoil(6) = par_v(6)
				par_vclgcoil(7) = par_v(7)
				par_vclgcoil(8) = par_v(12)
				par_vclgcoil(9) = par_v(13)
				par_vclgcoil(10)= par_v(14)
				par_vclgcoil(11)= par_v(15)
				par_vclgcoil(12)= par_v(16)
				par_vclgcoil(13)= par_v(17)
				par_vclgcoil(14)= par_v(18)
				par_vclgcoil(15)= par_v(19)
				par_vclgcoil(16)= 0.0
				par_vclgcoil(17)= par_v(20)
				par_vclgcoil(18)= par_v(21) 
				par_vclgcoil(19)= par_v(22)
				par_vclgcoil(20)= par_v(23)
				par_vclgcoil(21)= par_v(24)
				par_vclgcoil(22)= par_v(25)
				par_vclgcoil(23)= par_v(26)

!*** Call cooling coil subroutine  
                CALL TYPE523(xinclgcoil,youtclgcoil,par_vclgcoil,saved_vclgcoil,iostatclgcoil)

!*** Assign output values
				yout(1)  = youtmb(1)
				yout(2)  = youtmb(2)
				yout(3)  = youtclgcoil(2)
				yout(4)  = youtclgcoil(3)

				yout(5)  = youtclgcoil(1)
				yout(6)  = youtclgcoil(5)
				yout(7)  = youtclgcoil(6)
				yout(8)  = youtclgcoil(7)
				yout(9)  = youtclgcoil(8)
				yout(10) = youtclgcoil(9)
				yout(11) = youtclgcoil(10)
				yout(12) = youtclgcoil(11)
				yout(13) = youtclgcoil(12)

				yout(14) = youtfan(1)
				yout(15) = TWIHTG
				yout(16) = PWOHTG
				yout(17) = 0.0
				yout(18) = -99.9
				yout(19) = 0.0
				yout(20) = 0.0
				yout(21) = 0.0
				yout(22) = 0.0
	                
		ELSEIF ((VSTEMCLG.EQ.0.0).AND.(VSTEMHTG.NE.0.0)) THEN
!*** Read in heating coil inputs
				xinhtgcoil(1)= youtfan(1)
				TAIHTG       = youtfan(1)
			    GIHTG        = youtmb(2)

!*** Limit humidity ratio to saturated value
				GSAT         = FWPHI(TAIHTG,100.0,PATMPA)
				GIHTG        = MIN(GIHTG,GSAT)
				xinhtgcoil(2)= GIHTG
				xinhtgcoil(3)= 0.0
				xinhtgcoil(4)= youtmb(3)
				xinhtgcoil(5)= TWIHTG
                xinhtgcoil(6)= WWPRIMHTG
                xinhtgcoil(7)= PWOHTG 
	            xinhtgcoil(8)= VSTEMHTG
				xinhtgcoil(9)= TSDYNHTG
!*** Read in heating coil parameters
				par_vhtgcoil(1) = par_v(1)
				par_vhtgcoil(2) = par_v(2)
	            par_vhtgcoil(3) = par_v(3)
                par_vhtgcoil(4) = par_v(8)
				par_vhtgcoil(5) = par_v(9)
				par_vhtgcoil(6) = par_v(10)
				par_vhtgcoil(7) = par_v(11)
				par_vhtgcoil(8) = par_v(12)
				par_vhtgcoil(9) = par_v(13)
				par_vhtgcoil(10)= par_v(14)
				par_vhtgcoil(11)= par_v(15)
				par_vhtgcoil(12)= par_v(16)
				par_vhtgcoil(13)= par_v(17)
				par_vhtgcoil(14)= par_v(18)
				par_vhtgcoil(15)= par_v(19)
				par_vhtgcoil(16)= 0.0
				par_vhtgcoil(17)= par_v(27)
				par_vhtgcoil(18)= par_v(28) 
				par_vhtgcoil(19)= par_v(29)
				par_vhtgcoil(20)= par_v(30)
				par_vhtgcoil(21)= par_v(31)
				par_vhtgcoil(22)= par_v(32)
				par_vhtgcoil(23)= par_v(33)

!*** Call heating coil subroutine  
				CALL TYPE523(xinhtgcoil,youthtgcoil,par_vhtgcoil,saved_vhtgcoil,iostathtgcoil)
!*** Assign output values
                yout(1)  = youtmb(1)
				yout(2)  = youtmb(2)
				yout(3)  = youthtgcoil(2)
				yout(4)  = youthtgcoil(3)

				yout(5)  = youtfan(1)
				yout(6)  = TWICLG
				yout(7)  = PWOCLG
				yout(8)  = 0.0
				yout(9)  = -99.9
				yout(10) = 0.0
				yout(11) = 0.0
				yout(12) = 0.0
				yout(13) = 0.0

				yout(14) = youthtgcoil(1)
				yout(15) = youthtgcoil(5)
				yout(16) = youthtgcoil(6)
				yout(17) = youthtgcoil(7)
				yout(18) = youthtgcoil(8)
				yout(19) = youthtgcoil(9)
				yout(20) = youthtgcoil(10)
				yout(21) = youthtgcoil(11)
				yout(22) = youthtgcoil(12)
										
		ELSEIF ((VSTEMCLG.EQ.0.0).AND.(VSTEMHTG.EQ.0.0)) THEN
		    
!*** Assign output valuses when FCU is off or in economizer mode			    
				yout(1)  = youtmb(1)
				yout(2)  = youtmb(2)
				yout(3)  = youtfan(1)
				yout(4)  = youtmb(2)

				yout(5)  = youtfan(1)
				yout(6)  = TWICLG
				yout(7)  = PWOCLG
				yout(8)  = 0.0
				yout(9)  = -99.9
				yout(10) = 0.0
				yout(11) = 0.0
				yout(12) = 0.0
				yout(13) = 0.0

				yout(14) = youtfan(1)
				yout(15) = TWIHTG
				yout(16) = PWOHTG
				yout(17) = 0.0
				yout(18) = -99.9
				yout(19) = 0.0
				yout(20) = 0.0
				yout(21) = 0.0
				yout(22) = 0.0		
			
		ENDIF
         
!*** Allow freezing of algebraic variables
!      do i=i,no
!            iostat(i)=1
!      enddo
!*** Return
      return
      end subroutine type313

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motor-driven actuator
! *
! * PURPOSE:        Calculate the position of a constant speed,
! *                 motor-driven, actuator, accounting for hysteresis,
! *                 crank geometry and range mismatch. Reverse action
! *                 and "stuck" faults are also treated.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. posdem  : demanded position                                  (-)
! *
! * OUTPUTS
! * =======
! *  1. poscoel : position of final control element                  (-)
! *  2. posmot  : motor position                                     (-)
! *  3. tssrev  : number of stop/starts/reversals                    (-)
! *  4. ttrav   : total distance travelled by final control element  (-)
! *
! * PARAMETERS
! * ==========
! *  1. directn : 1=forward, -1=reverse, 0=stuck
! *  2. startpos: starting position (0-1)
! *  3. ttran   : travel time (lim-lim) (s)
! *  4. restart : minimum change in demanded position for movement   (-)
! *  5. hys     : hysteresis (-)
! *  6. crang   : crank travel angle (0 for linear)
! *  7. poscoelu: upper limit of control element range on actuator scale (-)
! *  8. poscoell: lower limit of control element range on actuator scale (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call
! *  2. posmot  : motor position at previous call
! *  3. posmotp : motor position at previous timestep
! *  4. posdem  : demanded position at previous call
! *  5. posdemp : demanded position at previous timestep
! *  6. modir   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. modirp  : +1=forward, -1=backward, 0=stopped at previous step time
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev step time
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous step time
! * 12. posact  : position of actuator at previous call
! * 13. posactp : position of actuator at previous step time
! * 14. poscoel : position of final control element at previous call
! * 15. poscoelp: position of final control element at previous step time
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes actuator stops when position error is
!                        exactly zero. Assumes motor starts and stops
!                        instanteously. Assumes demanded position has
!                        the same value throughout the time-step, i.e.
!                        the value of the input changes at the beginning
!                        of the time-step and remains valid through the
!                        time-step.
! 
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                June 16, 1994
!   LAST MODIFIED:       January 18, 1996
! 
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTRSIS
!
!   REVISION HISTORY:    None
! 
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
! 
! **********************************************************************
! * INTERNAL VAIABLES
! * =================
! * quick    : true if actuator moves essentially instantaneously
! * dposmax  : maximum distance actuator can travel in one time-step
! * dposdem  : difference between demanded and current positions at
! *            start of new time-step
! * dposdemp : difference between demanded and current positions at
! *            start of previous time-step
! * modirp   : motor direction at start of previous time-step
! * modire   : motor direction at end of previous time-step
! * modir    : motor direction at start of new time-step
! * poscrank : linear position of crank
! * posact   : position of actuator after linkage slack
! * con      : true if demanded position constant (frozen or fixed boundary
! *            condition) (HVACSIM+ only)
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type321(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=4,np=8,ns=15
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        logical      :: con
        integer      :: directn
        logical      :: quick
        real         :: dtr=0.01745,small=1.e-6
        real         :: posdem,startpos,ttran,restart,hys,crang,&
                        poscoelu,poscoell,posmotp,posdemp,tssrevp,&
                        ttravp,posactp,poscoelp,posmot,tssrev,dposmax, &
                        dposdemp,dposdem,range,poscrank,posact,poscoel,&
                        span,ttrav
        integer      :: i,is,modirp,modire,modirdem,modir

        quick = .true.      ! added   12/6/1999

! **** Read in inputs
        posdem   = xin(1)
		
! **** Read in parameters
        directn  = nint(par_v(1))
        startpos = par_v(2)
        if (startpos<0.0 .or. startpos>1.0) then
            stop 'type 321: starting position out of range'
        endif
        ttran    = par_v(3)
        restart  = par_v(4)
        hys      = par_v(5)
        crang    = par_v(6)
        poscoelu = par_v(7)
        poscoell = par_v(8)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.0
            endif
            if (init==0) then
                saved_v(2) = startpos
                saved_v(4) = startpos
                do is = 6,ns-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - Update previous sample instant values
            do is=2,ns-1,2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Previous values
        posmotp  = saved_v(3)
        posdemp  = saved_v(5)
        modirp   = nint(saved_v(7))
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        posactp  = saved_v(13)
        poscoelp = saved_v(15)
! **** Limit control signal and invert if reverse acting
        posdem=max(0.,min(posdem,1.))
        if (directn==-1) posdem=1.-posdem
! **** Determine curent position based on situation at previous step time
        if (modirp==0) then
! **** Motor off throughout previous time-step
            posmot = posmotp
            modire = 0
            tssrev = tssrevp
        else
! **** Motor on at beginning of previous time-step - Determine if demanded
! **** Position attained
            if (ttran<=(tstep*small)) then
                quick   = .true.
            else
                quick   = .false.
                dposmax = tstep/ttran
            endif
! **** Demanded change in position during previous time-step
            dposdemp = posdemp-posmotp
            if (quick .or. abs(dposdemp)<=dposmax) then
! **** Demanded position attained - Actuator stopped moving at some point
! **** during previous time-step
                posmot = posdemp
                modire = 0.0
                tssrev = tssrevp+1.
            else
! **** Demanded position not attained - Actuator was on continuously
                posmot = posmotp+sign(dposmax,dposdemp)
                modire = nint(sign(1.,dposdemp))
                tssrev = tssrevp
            endif
        endif
! **** Determine response to current control signal
        dposdem  = posdem-posmot
! **** State of motor required to minimise position error
        if (dposdem==0.0) then
            modirdem = 0
        else
            modirdem = nint(sign(1.,dposdem))
        endif
! **** First consider case of motor not in desired state
        if (modire/=modirdem) then
! **** Motor was off or moving in `wrong' direction - Determine if motor starts
            if (abs(dposdem)>restart) then
! **** Demanded and actual positions sufficiently different to start motor
                modir = modirdem
                if (modir/=modire) then
! **** Actuator motion different from that at end of previous timestep
                    tssrev = tssrev+1.
                else
! *****Actuator motion unchanged from previous timestep
                    tssrev = tssrev
                endif
            else
! **** Within deadband - Motor stays off
                modir  = 0
                tssrev = tssrev
            endif
        else
! **** Motor in desired state at end of previous timestep - No change in
! **** state
            modir  = modire
            tssrev = tssrev
        endif
        if (directn==0) then
! **** Stuck/disconnected - Set to initial position
            posmot = startpos
        endif
! **** Non-linearity due to crank, and hysteresis
        if (crang>0.0) then
            range    = 2.*sin(dtr*crang/2.)
            poscrank = 0.5+sin(dtr*crang*(posmot-0.5))/range
        else
            poscrank = posmot
        endif
! **** Hysteresis due to slack in linkage
        call hystrsis(poscrank,posactp,hys,posact)
! **** Range mismatch - linear transformation, then limit to range 0-1
        poscoel = span(posact,poscoell,poscoelu,0.0,1.0)
! **** Distance travelled by control element
        ttrav   = ttravp+abs(poscoel-poscoelp)
! **** Save time of current call
        saved_v(1)=time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = posmot
        saved_v(4)  = posdem
        saved_v(6)  = float(modir)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = posact
        saved_v(14) = poscoel
! **** Output
        yout(1) = poscoel
        yout(2) = posmot
        yout(3) = tssrev
        yout(4) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if position error small and demanded position
! **** constant or if actuator responds instantly
        con = (iostat(1)<-1).or.(iostat(1)==2)
        if ((abs(dposdem)<=max(restart,rtolx).and.con).or.quick) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 0
            enddo
        endif
! **** Return
        return
        end subroutine type321

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motor-driven actuator with fault
! *
! * PURPOSE:        Calculate the position of a constant speed,
! *                 motor-driven, actuator, accounting for hysteresis,
! *                 crank geometry and range mismatch. Reverse action
! *                 and "stuck" faults are also treated.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. posdem  : demanded position                                  (-)
! *
! * OUTPUTS
! * =======
! *  1. poscoel : position of final control element                  (-)
! *  2. posmot  : motor position                                     (-)
! *  3. tssrev  : number of stop/starts/reversals                    (-)
! *  4. ttrav   : total distance travelled by final control element  (-)
! *
! * PARAMETERS
! * ==========
! *  1. directn : 1=forward, -1=reverse, 0=stuck
! *  2. startpos: starting position (0-1)
! *  3. ttran   : travel time (lim-lim) (s)
! *  4. restart : minimum change in demanded position for movement       (-)
! *  5. hys     : hysteresis (-)
! *  6. crang   : crank travel angle (0 for linear)
! *  7. poscoelu: upper limit of control element range on actuator scale (-)
! *  8. poscoell: lower limit of control element range on actuator scale (-)
! *******************************DREXEL ADDED************************************
! *  9. Location: 1-cooling valve actuator,2-heating valve actuator 
! *               3-damper actuator,                                     (-)
! *******************************DREXEL ADDED************************************
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call
! *  2. posmot  : motor position at previous call
! *  3. posmotp : motor position at previous timestep
! *  4. posdem  : demanded position at previous call
! *  5. posdemp : demanded position at previous timestep
! *  6. modir   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. modirp  : +1=forward, -1=backward, 0=stopped at previous step time
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev step time
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous step time
! * 12. posact  : position of actuator at previous call
! * 13. posactp : position of actuator at previous step time
! * 14. poscoel : position of final control element at previous call
! * 15. poscoelp: position of final control element at previous step time
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes actuator stops when position error is
!                        exactly zero. Assumes motor starts and stops
!                        instanteously. Assumes demanded position has
!                        the same value throughout the time-step, i.e.
!                        the value of the input changes at the beginning
!                        of the time-step and remains valid through the
!                        time-step.
! 
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!   REVISED BY:          Shokouh Pourarian, Jin Wen
!                        Drexel University
! 
!   DATE:                June 16, 1994
!   LAST MODIFIED:       January 18, 1996
! 
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTRSIS
!
!   REVISION HISTORY:    None
! 
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
! 
! **********************************************************************
! * INTERNAL VAIABLES
! * =================
! * quick    : true if actuator moves essentially instantaneously
! * dposmax  : maximum distance actuator can travel in one time-step
! * dposdem  : difference between demanded and current positions at
! *            start of new time-step
! * dposdemp : difference between demanded and current positions at
! *            start of previous time-step
! * modirp   : motor direction at start of previous time-step
! * modire   : motor direction at end of previous time-step
! * modir    : motor direction at start of new time-step
! * poscrank : linear position of crank
! * posact   : position of actuator after linkage slack
! * con      : true if demanded position constant (frozen or fixed boundary
! *            condition) (HVACSIM+ only)
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type331(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=4,np=9,ns=15
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        logical      :: con
        integer      :: directn
        logical      :: quick
        real         :: dtr=0.01745,small=1.e-6
        real         :: posdem,startpos,ttran,restart,hys,crang,&
                        poscoelu,poscoell,posmotp,posdemp,tssrevp,&
                        ttravp,posactp,poscoelp,posmot,tssrev,dposmax, &
                        dposdemp,dposdem,range,poscrank,posact,poscoel,&
                        span,ttrav
        integer      :: i,is,modirp,modire,modirdem,modir,location

        quick = .true.      ! added   12/6/1999

! **** Read in inputs
        posdem   = xin(1)
		
! **** Read in parameters
        directn  = nint(par_v(1))
        startpos = par_v(2)
        if (startpos<0.0 .or. startpos>1.0) then
            stop 'type 321: starting position out of range'
        endif
        ttran    = par_v(3)
        restart  = par_v(4)
        hys      = par_v(5)
        crang    = par_v(6)
        poscoelu = par_v(7)
        poscoell = par_v(8)
! *******************************DREXEL ADDED************************************
		location = par_v(9)
!		location = int(location)
		
! **** Fault flag on
	
		if (location.EQ.1)then
			if (CONcoolreverse.EQ.0) then
				posdem = posdem
			elseif (CONcoolreverse.EQ.1) then
				posdem = 1 - posdem
			endif
        elseif (location.EQ.2)then
			if (CONheatreverse.EQ.0) then
				posdem = posdem
			elseif (CONheatreverse.EQ.1) then
				posdem = 1 - posdem
			endif
        endif
! **** Fault flag off
		
! *******************************DREXEL ADDED************************************ 
		
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.0
            endif
            if (init==0) then
                saved_v(2) = startpos
                saved_v(4) = startpos
                do is = 6,ns-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - Update previous sample instant values
            do is=2,ns-1,2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Previous values
        posmotp  = saved_v(3)
        posdemp  = saved_v(5)
        modirp   = nint(saved_v(7))
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        posactp  = saved_v(13)
        poscoelp = saved_v(15)
! **** Limit control signal and invert if reverse acting
        posdem=max(0.,min(posdem,1.))
        if (directn==-1) posdem=1.-posdem
! **** Determine curent position based on situation at previous step time
        if (modirp==0) then
! **** Motor off throughout previous time-step
            posmot = posmotp
            modire = 0
            tssrev = tssrevp
        else
! **** Motor on at beginning of previous time-step - Determine if demanded
! **** Position attained
            if (ttran<=(tstep*small)) then
                quick   = .true.
            else
                quick   = .false.
                dposmax = tstep/ttran
            endif
! **** Demanded change in position during previous time-step
            dposdemp = posdemp-posmotp
            if (quick .or. abs(dposdemp)<=dposmax) then
! **** Demanded position attained - Actuator stopped moving at some point
! **** during previous time-step
                posmot = posdemp
                modire = 0.0
                tssrev = tssrevp+1.
            else
! **** Demanded position not attained - Actuator was on continuously
                posmot = posmotp+sign(dposmax,dposdemp)
                modire = nint(sign(1.,dposdemp))
                tssrev = tssrevp
            endif
        endif
! **** Determine response to current control signal
        dposdem  = posdem-posmot
! **** State of motor required to minimise position error
        if (dposdem==0.0) then
            modirdem = 0
        else
            modirdem = nint(sign(1.,dposdem))
        endif
! **** First consider case of motor not in desired state
        if (modire/=modirdem) then
! **** Motor was off or moving in `wrong' direction - Determine if motor starts
            if (abs(dposdem)>restart) then
! **** Demanded and actual positions sufficiently different to start motor
                modir = modirdem
                if (modir/=modire) then
! **** Actuator motion different from that at end of previous timestep
                    tssrev = tssrev+1.
                else
! *****Actuator motion unchanged from previous timestep
                    tssrev = tssrev
                endif
            else
! **** Within deadband - Motor stays off
                modir  = 0
                tssrev = tssrev
            endif
        else
! **** Motor in desired state at end of previous timestep - No change in
! **** state
            modir  = modire
            tssrev = tssrev
        endif
        if (directn==0) then
! **** Stuck/disconnected - Set to initial position
            posmot = startpos
        endif
! **** Non-linearity due to crank, and hysteresis
        if (crang>0.0) then
            range    = 2.*sin(dtr*crang/2.)
            poscrank = 0.5+sin(dtr*crang*(posmot-0.5))/range
        else
            poscrank = posmot
        endif
! **** Hysteresis due to slack in linkage
        call hystrsis(poscrank,posactp,hys,posact)
! **** Range mismatch - linear transformation, then limit to range 0-1
        poscoel = span(posact,poscoell,poscoelu,0.0,1.0)
! **** Distance travelled by control element
        ttrav   = ttravp+abs(poscoel-poscoelp)
! **** Save time of current call
        saved_v(1)=time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = posmot
        saved_v(4)  = posdem
        saved_v(6)  = float(modir)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = posact
        saved_v(14) = poscoel
! **** Output
        yout(1) = poscoel
        yout(2) = posmot
        yout(3) = tssrev
        yout(4) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if position error small and demanded position
! **** constant or if actuator responds instantly
        con = (iostat(1)<-1).or.(iostat(1)==2)
        if ((abs(dposdem)<=max(restart,rtolx).and.con).or.quick) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 0
            enddo
        endif
! **** Return
        return
        end subroutine type331

! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Damper
! *
! * PURPOSE:    Calculates the pressure drop across a damper from the
! *             mass flow rate. Uses a linear relationship between the
! *             position angle of the blade(s) (THETA) and the logarithm
! *             of the loss coefficient in the range 15 deg < THETA <
! *             55 deg (opposed/single) or 65 deg (parallel). Uses
! *             quadratic interpolation functions in the ranges
! *             0 < THETA < 15 deg and 55/65 deg < THETA < 90 deg.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. w       : Mass flow rate                                  (kg/s)
! *  2. pout    : Outlet pressure                                  (kPa)
! *  3. c       : Damper position (0=closed, 1=open)                 (-)
! *
! * OUTPUTS
! * =======
! *  1. pin     : Inlet pressure                                   (kPa)
! *
! * PARAMETERS
! * ==========
! *  1. ipar    : Damper geometry: opposed/single (0) or parallel (1)(-)
! *  2. ropen   : Open resistance                            (0.001/k.m)
! *  3. farea   : Face area                                         (m2)
! *  4. fleak   : Leakage (fraction of full flow)                    (-)
! *  5. a       : a coefficient in ln(K)=a+b*THETA (-1 for Legg)     (-)
! *  6. b       : b coefficient in ln(K)=a+b*THETA (-1 for Legg)     (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Uses fixed limits on range of validity of
!                        logoarithmic relationship
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 8, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RDAMPER
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           825-RP Final Report
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * dp      : pressure drop across fresh air branch
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type322(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=3,no=1,np=6,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
        real         :: w,pout,c,ropen,farea,fleak,ax,b,r,rdamper,dp,&
                        dpqudlin,pin
        integer      :: i,ipar

! **** Read in inputs
        w    = xin(1)
        pout = xin(2)
        c    = xin(3)
! **** Read and check parameters
        ipar    = nint(par_v(1))
        ropen   = par_v(2)
        if (ropen<=0.0) stop 'type322: ropen not greater than 0'
        farea   = par_v(3)
        fleak   = par_v(4)
        ax      = par_v(5)
! **** Default is value recommended by legg
        if (ax<0.0) ax = alegg(ipar)
        b       = par_v(6)
! **** Default is value recommended by legg
        if (b<0.0) b = blegg(ipar)
! **** Calculate resistance of damper
        r = rdamper(c,ropen,fleak,farea,ax,b,ipar)
! **** Calculate pressure drop and inlet pressure
        dp = dpqudlin(r,w)
        pin = pout+dp
! **** Assign outputs
        yout(1) = pin
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo
        return
        end subroutine type322
! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Damper - calculates flow rate
! *
! * PURPOSE:    Calculates the mass flow rate through a damper from the
! *             presuure drop. Uses a linear relationship between the
! *             position angle of the blade(s) (THETA) and the logarithm
! *             of the loss coefficient in the range 15 deg < THETA <
! *             55 deg (opposed/single) or 65 deg (parallel). Uses
! *             quadratic interpolation functions in the ranges
! *             0 < THETA < 15 deg and 55/65 deg < THETA < 90 deg.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. pin     : Inlet pressure                                   (kPa)
! *  2. pout    : Outlet pressure                                  (kPa)
! *  3. c       : Damper position (0=closed, 1=open)                 (-)
! *
! * OUTPUTS
! * =======
! *  1. w       : Mass flow rate                                  (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. ipar    : Damper geometry: opposed/single (0) or parallel (1)(-)
! *  2. ropen   : Open resistance                            (0.001/k.m)
! *  3. farea   : Face area                                         (m2)
! *  4. fleak   : Leakage (fraction of full flow)                    (-)
! *  5. a       : a coefficient in ln(K)=a+b*THETA (-1 for Legg)     (-)
! *  6. b       : b coefficient in ln(K)=a+b*THETA (-1 for Legg)     (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Uses fixed limits on range of validity of
!                        logoarithmic relationship
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 8, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RDAMPER
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           825-RP Final Report
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * dp      : pressure drop across fresh air branch
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type323(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=3,no=1,np=6,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
        real         :: pin,pout,c,ropen,farea,fleak,ax,b,r,rdamper,&
                        dp,w,wqudlin
        integer      :: i,ipar

! **** Read in inputs
        pin  = xin(1)
        pout = xin(2)
        c    = xin(3)
! **** Read and check parameters
        ipar    = nint(par_v(1))
        ropen   = par_v(2)
        if (ropen<=0.0) stop 'type323: ropen not greater than 0'
        farea   = par_v(3)
        fleak   = par_v(4)
        ax      = par_v(5)
! **** Default is value recommended by legg
        if (ax<0.0) ax = alegg(ipar)
        b       = par_v(6)
! **** Default is value recommended by legg
        if (b<0.0) b = blegg(ipar)
! **** Calculate resistance of damper
        r = rdamper(c,ropen,fleak,farea,ax,b,ipar)
! **** Calculate pressure drop and inlet pressure
        dp = pin-pout
        w = wqudlin(r,dp)
! **** Assign output
        yout(1) = w
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo

        return
        end subroutine type323
! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Mixing box (implicit flow)
! *
! * PURPOSE:    Calculates the mixed air temperature and humidity and
! *             the mixed air and extract air pressures. Uses Legg's
! *             correlations for the resistance of parallel and opposed
! *             blade dampers
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tout    : Outside air dry bulb temperature                   (C)
! *  2. gout    : Outside air humidity ratio                     (kg/kg)
! *  3. text    : Extract air dry bulb temperature                   (C)
! *  4. gext    : Extract air humidity ratio                     (kg/kg)
! *  5. pout    : Outside air intake pressure                      (kPa)
! *  6. pexh    : Exhaust air outlet pressure                      (kPa)
! *  7. wmix    : Mixed air mass flow rate                        (kg/s)
! *  8. wext    : Extract air mass flow rate                      (kg/s)
! *  9. cout    : Outside air damper position (0=closed, 1=open)     (-)
! * 10. crec    : Recirc air damper position (0=open if PAR(16)=0)   (-)
! * 11. cexh    : Exhaust air damper position (0=closed, 1=open)     (-)
! *
! * OUTPUTS
! * =======
! *  1. tmix    : Mixed air dry bulb temperature                     (C)
! *  2. gmix    : Mixed air humidity ratio                       (kg/kg)
! *  3. pmix    : Mixed air pressure                               (kPa)
! *  4. pext    : Extract air pressure                             (kPa)
! *  5. wout    : Outside air mass flow rate                      (kg/s)
! *  6. wrec    : Recirc air mass flow rate                       (kg/s)
! *  7. wexh    : Exhaust air mass flow rate                      (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. iparout : Outside air damper: opposed (0) or parallel (1)    (-)
! *  2. iparrec : Recirc air damper: opposed (0) or parallel (1)     (-)
! *  3. iparexh : Exhaust air damper: opposed (0) or parallel (1)    (-)
! *  4. ropenout: Open resist. of outside air damper         (0.001/k.m)
! *  5. ropenrec: Open resist. of recirc air damper          (0.001/k.m)
! *  6. ropenexh: Open resist. of exhaust air damper         (0.001/k.m)
! *  7. fareaout: Face area of outside air damper                   (m2)
! *  8. farearec: Face area of recirc air damper                    (m2)
! *  9. fareaexh: Face area of exhaust air damper                   (m2)
! * 10. fleakout: Leakage for outside air damper (fraction of full flow)
! * 11. fleakrec: Leakage for recirc air damper  (fraction of full flow)
! * 12. fleakexh: Leakage for exhaust air damper (fraction of full flow)
! * 13. rfixout : Fixed resistance in outside air branch     (0.001/k.m)
! * 14. rfixrec : Fixed resistance in recirc air branch      (0.001/k.m)
! * 15. rfixexh : Fixed resistance in exhaust air branch     (0.001/k.m)
! * 16. noninver: 0=invert recirc air damper, 1=not inverted         (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes positive mixed and extract flow rates
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 8, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWSPLT, MOISTMIX
!   FUNCTIONS  CALLED:   None
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * rout    : total resistance of outside air branch
! * rrec    : total resistance of recirculation branch
! * rexh    : total resistance of exhaust branch
! * dpexhout: difference between exhaust and outside pressures
! * dpout   : pressure drop across outside branch
! * dprec   : pressure drop across recirculation branch
! * dpexh   : pressure drop across exhaust branch
! * srout   : "signed" resistance of outside branch
! * srrec   : "signed" resistance of recirculation branch
! * srexh   : "signed" resistance of exhaust branch
! * diff    : (b^2 - 4*a*c)/(2*a)
! * wxxxp   : flow in xxx branch corresponding to positive root
! * wxxxn   : flow in xxx branch corresponding to negative root
! * rootpos : true if positive root consistent with known directions
! * rootneg : true if negative root consistent with known directions
! * wsmall  : threshold for significant flow rate
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type324(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=11,no=7,np=16,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(20)    :: t,gx,w
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
        logical      :: rootpos,rootneg
        real         :: wsmall=1.0e-6
        real         :: ropenout,ropenrec,ropenexh,fareaout,farearec,&
                        fareaexh,fleakout,fleakrec,fleakexh,rfixout,&
                        rfixrec,rfixexh,rout,rrec,rexh,dpexhout,&
                        dpout,srout,dprec,srrec,dpexh,srexh,wout,wrec,&
                        wexh,aa,bb,cc,bbb,ccc,bbbsq,diff,z,wrecp,wrecn,&
                        wexhp,woutp,wexhn,woutn,pmix,pext,tmix,gmix,&
                        wexhrev,wdummy,grec,trec,tout,gout,text,gext,&
                        pout,pexh,wmix,wext,cout,crec,cexh,rdamper
        integer      :: i,iparrec,iparexh,iparout,noninver

! **** Read and check parameters
        iparout = nint(par_v(1))
        iparrec = nint(par_v(2))
        iparexh = nint(par_v(3))
        ropenout= par_v(4)
        ropenrec= par_v(5)
        ropenexh= par_v(6)
        fareaout= par_v(7)
        farearec= par_v(8)
        fareaexh= par_v(9)
        fleakout= par_v(10)
        fleakrec= par_v(11)
        fleakexh= par_v(12)
        rfixout= par_v(13)
        rfixrec= par_v(14)
        rfixexh= par_v(15)
        noninver = nint(par_v(16))

! **** Read in inputs
        tout = xin(1)                 !  Moved on Feb. 6, 2007
        gout = xin(2)
        text = xin(3)
        gext = xin(4)
        pout = xin(5)
        pexh = xin(6)
        wmix = xin(7)
        wext = xin(8)
        cout = xin(9)
        if (noninver/=1) then
            crec = 1.0-xin(10)
        else
            crec = xin(10)
        endif
        cexh = xin(11)

! **** Calculate resistance of each branch
! **** Resistance is sum of damper resistance and fixed resistance
        rout = rdamper(cout,ropenout,fleakout,fareaout,alegg(iparout),&
                       blegg(iparout),iparout) + rfixout
        if (rout<=0.0) stop &
            'type324: resistance in outside branch not greater than 0'
        rrec = rdamper(crec,ropenrec,fleakrec,farearec,alegg(iparrec),&
                       blegg(iparrec),iparrec) + rfixrec
        if (rrec<=0.0) stop &
            'type324: resistance in recirc branch not greater than 0'
        rexh = rdamper(cexh,ropenexh,fleakexh,fareaexh,alegg(iparexh),&
                       blegg(iparexh),iparexh) + rfixexh
        if (rexh<=0.0) stop &
            'type324: resistance in exhaust branch not greater than 0'
! **** Calculate "signed" resistances (see printed documentation)
! **** Test for direction of the outside, return and exhaust flows by
! **** determining the sign of the pressure difference across the
! **** corresponding resistance assuming the flow in question is zero.
        dpexhout = pexh - pout
! **** Outside:
        dpout = rrec*wmix*abs(wmix) - rexh*(wext-wmix)*abs(wext-wmix) -&
                dpexhout
        srout = sign(rout,dpout)
! **** Return:
        dprec = rexh*wext*abs(wext) + rout*wmix*abs(wmix) + dpexhout
        srrec = sign(rrec,dprec)
! **** Exhaust:
        dpexh = rrec*wext*abs(wext) - rout*(wmix-wext)*abs(wmix-wext) -&
                dpexhout
        srexh = sign(rexh,dpexh)
! **** Determine necessity of solving quadratic - Check for special cases
        if (srout==0) then
! **** Outside flow zero
            wout = 0.0
            wrec = wmix
            wexh = wext - wrec
        elseif (srrec==0) then
! **** Recirculation flow zero
            wrec = 0.0
            wout = wmix
            wexh = wext
        elseif (srexh==0) then
! **** Exhaust flow zero
            wexh = 0.0
            wrec = wext
            wout = wmix -wrec
        else
! **** All three unknown flows non-zero:
! **** Calc wrec by solving a quadratic of the form a*wrec**2 + b*wrec + c = 0
! **** Note that aa=a, bb=-b/2, cc=c,  bbb=-b/2a,  ccc=c/a
            aa   = srout+srexh-srrec
            bb   = srout*wmix+srexh*wext
            cc   = srout*wmix*wmix+srexh*wext*wext+dpexhout
            if (aa==0.) then
! **** One root
                wrec = cc/(2.0*bb)
            else
! **** Two roots
                bbb = bb/aa
                ccc = cc/aa
                bbbsq = bbb*bbb
                diff = bbbsq-ccc
                if (diff>0) then
! **** Roots unequal
                    z = sqrt(diff)
                    wrecp = bbb + z
                    wrecn = bbb - z
! **** Check each root for consistency with directions of air flow
! **** already determined
                    wexhp = wext - wrecp
                    woutp = wmix - wrecp
                    if (wexhp*srexh<0.0 .or.&
                        wrecp*srrec<0.0 .or.&
                        woutp*srout<0.0) then
                        rootpos = .false.
                    else
                        rootpos = .true.
                    endif
                    wexhn = wext-wrecn
                    woutn = wmix-wrecn
                    if (wexhn*srexh<0.0 .or.&
                        wrecn*srrec<0.0 .or.&
                        woutn*srout<0.0) then
                        rootneg = .false.
                    else
                        rootneg = .true.
                    endif
                    if (.not.rootpos .and. .not.rootneg) then
                        stop 'type324: no consistent solution'
                    elseif (rootpos .and. rootneg) then
                           stop 'type324: ambiguous solution'
                       elseif (rootpos .and. .not.rootneg) then
                           wrec = wrecp
                       elseif (rootneg .and. .not.rootpos) then
                           wrec = wrecn
                    else
                        stop 'type324: logic fault 1'
                    endif
                elseif (diff==0) then
! **** Equal roots
                    wrec = bbb
                else
! **** Complex roots (coding error, negative resistances ...?)
                    stop  'type 324: complex roots'
                endif
            endif
! **** Calculate remaining unknown flows
            wout = wmix - wrec
            wexh = wext - wrec
! **** Calculate mixed and extract pressures
            pmix = pout - srout*wout*wout
            pext = pexh + srexh*wexh*wexh
        endif
! **** Having calculated flows, calculate mixed temperature and humidity
! **** Model assumes wmix and wext both non-negative. warn if not true.
        if (lastit==1) then
            if (wmix<-wsmall) write(*,*) 'type 324: wmix < ',-wsmall
            if (wext<-wsmall) write(*,*) 'type 324: wext < ',-wsmall
        endif
! **** Check for zero or reverse mixed flow
        if (wmix<=wsmall) then
! **** Reverse mixed flow, take mixed temperature and humidity ratio to be
! **** an average of the outside and extract conditions so as to provide a
! **** reasonable guess for subsequent iterations
            tmix = (tout+text)/2.0
            gmix = (gout+gext)/2.0
        elseif (wmix>wsmall .and. wext<=wsmall) then
! **** Zero/reverse extract flow - Mixed condition is the same as outside
! **** condition
            tmix = tout
            gmix = gout
        elseif (wmix>wsmall .and. wext>wsmall ) then
! **** Normal, forward mixed and extract flows - Calculate mixed conditions
! **** first calculate recirc air conditions if recirc flow positive
            if (wrec>wsmall) then
! **** Test for zero/reverse exhaust flow
                if (wexh<wsmall) then
! **** Reverse exhaust flow - Recirc flow is a mixture of extract and outside
! **** conditions
                    wexhrev = -wexh
                    t(1) = text
                    gx(1) = gext
                    w(1) = wext
                    t(2) = tout
                    gx(2) = gout
                    w(2) = wexhrev
                    call moistmix(t,gx,w,2,trec,grec,wdummy)
                else
! **** Normal, forward exhaust flow - recirc condition is extract condition
                    trec = text
                    grec = gext
                endif
            endif
! **** Calculate mixed air conditions
! **** mixed condition is a combination of the recirc and outside conditions
! **** moistmix also tests for reverse flows and sets the mixed condition
! **** to the forward flow condition, or to an average of the flow conditions
! **** if both are reverse
            t(1) = trec
            gx(1) = grec
            w(1) = wrec
            t(2) = tout
            gx(2) = gout
            w(2) = wout
            call moistmix(t,gx,w,2,tmix,gmix,wdummy)
        else
            stop 'type324: logic fault 2'
        endif
! **** Assign outputs
        yout(1) = tmix
        yout(2) = gmix
        yout(3) = pmix
        yout(4) = pext
        yout(5) = wout
        yout(6) = wrec
        yout(7) = wexh
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo

        return
        end subroutine type324

! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Mixing box
! *
! * PURPOSE:    Calculates the mixed air flow rate, temperature and
! *             humidity and the extract air pressure. Uses Legg's
! *             correlations for the resistance of parallel and opposed
! *             blade dampers
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tri     : Outside air dry bulb temperature                   (C)
! *  2. gout    : Outside air humidity ratio                     (kg/kg)
! *  3. text    : Extract air dry bulb temperature                   (C)
! *  4. gext    : Extract air humidity ratio                     (kg/kg)
! *  5. pout    : Outside air intake pressure                      (kPa)
! *  6. pexh    : Exhaust air outlet pressure                      (kPa)
! *  7. pmix    : Mixed air outlet pressure                        (kPa)
! *  8. wext    : Extract air mass flow rate                      (kg/s)
! *  9. cout    : Outside air damper position (0=closed, 1=open)     (-)
! * 10. crec    : Recirc air damper position (0=open if PAR(16)=0)   (-)
! * 11. cexh    : Extract air damper position (0=closed, 1=open)     (-)
! *
! * OUTPUTS
! * =======
! *  1. tmix    : Mixed air dry bulb temperature                     (C)
! *  2. gmix    : Mixed air humidity ratio                       (kg/kg)
! *  3. wmix    : Mixed air mass flow rate                        (kg/s)
! *  4. pext    : Extract air pressure                             (kPa)
! *  5. wout    : Outside air mass flow rate                      (kg/s)
! *  6. wrec    : Recirc air mass flow rate                       (kg/s)
! *  7. wexh    : Exhaust air mass flow rate                      (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. iparout : Outside air damper: opposed (0) or parallel (1)    (-)
! *  2. iparrec : Recirc air damper: opposed (0) or parallel (1)     (-)
! *  3. iparexh : Exhaust air damper: opposed (0) or parallel (1)    (-)
! *  4. ropenout: Open resist. of outside air damper         (0.001/k.m)
! *  5. ropenrec: Open resist. of recirc air damper          (0.001/k.m)
! *  6. ropenexh: Open resist. of exhaust air damper         (0.001/k.m)
! *  7. fareaout: Face area of outside air damper                   (m2)
! *  8. farearec: Face area of recirc air damper                    (m2)
! *  9. fareaexh: Face area of exhaust air damper                   (m2)
! * 10. fleakout: Leakage for outside air damper  fraction of full flow)
! * 11. fleakrec: Leakage for recirc air damper  (fraction of full flow)
! * 12. fleakexh: Leakage for exhaust air damper (fraction of full flow)
! * 13. rfixout : Fixed resistance in outside air branch     (0.001/k.m)
! * 14. rfixrec : Fixed resistance in recirc air branch      (0.001/k.m)
! * 15. rfixexh : Fixed resistance in exhaust air branch     (0.001/k.m)
! * 16. noninver: 0=invert recirc air damper, 1=not inverted         (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes positive mixed and extract flow rates
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 8, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWSPLT, MOISTMIX
!   FUNCTIONS  CALLED:   None
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * rout    : total resistance of outside air branch
! * rrec    : total resistance of recirculation branch
! * rexh    : total resistance of exhaust branch
! * dp      : pressure drop across outside air branch
! * small   : outsideold for significant flow rate
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type325(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=11,no=7,np=16,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(20)    :: t,gx,w

        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel Added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        real,dimension(0:1)   :: aleggoa=(/-0.596,-1.51/),&
		                         bleggoa=(/0.1142,0.0842/)
		real,dimension(0:1)   :: aleggea=(/0.9633,-1.51/),&
                                 bleggea=(/0.0782,0.0842/)
		real,dimension(0:1)   :: aleggra=(/-1.8842,-1.51/),&
                                 bleggra=(/0.1338,0.0842/)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel Added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

        real         :: small=1.0e-10,wcritl=3.e-2,wcritu=6.e-2
        real         :: ropenout,ropenrec,ropenexh,fareaout,farearec,&
                        fareaexh,fleakout,fleakrec,fleakexh,rfixout,&
                        rfixrec ,rfixexh,tout,gout,text,gext,pout,&
                        pexh,pmix,wext,cout,crec,cexh,rout,rdamper,rrec,&
                        rexh,wexh,wrec,wexh1,wrec1,pext,dp,wout,wout1,wqudlin,wmix,tmix,&
                        gmix,wexhrev,wdummy,grec,trec
        integer      :: i,iparout,iparrec,iparexh,ifail,noninver

! **** Read and check parameters
        iparout = nint(par_v(1))
        iparrec = nint(par_v(2))
        iparexh = nint(par_v(3))
        ropenout= par_v(4)
        if (ropenout<=0.0) stop 'type325: ropenout not greater than 0'
        ropenrec= par_v(5)
        if (ropenrec<=0.0) stop 'type325: ropenrec not greater than 0'
        ropenexh= par_v(6)
        if (ropenexh<=0.0) stop 'type325: ropenexh not greater than 0'
        fareaout= par_v(7)
        farearec= par_v(8)
        fareaexh= par_v(9)
        fleakout= par_v(10)
        fleakrec= par_v(11)
        fleakexh= par_v(12)
        rfixout= par_v(13)
        if (rfixout<=0.0) stop 'type325: rfixout not greater than 0'
        rfixrec= par_v(14)
        if (rfixrec<=0.0) stop 'type325: rfixrec not greater than 0'
        rfixexh= par_v(15)
        if (rfixexh<=0.0) stop 'type325: rfixexh not greater than 0'
        noninver = nint(par_v(16))

! **** Read in inputs
        tout = xin(1)                ! Moved on Feb. 6, 2007
        gout = xin(2)
        text = xin(3)
        gext = xin(4)
        pout = xin(5)
        pexh = xin(6)
        pmix = xin(7)
        wext = xin(8)
        cout = xin(9)
        if (noninver/=1) then
            crec = 1.0-xin(10)
        else
            crec = xin(10)
        endif
        cexh = xin(11)
! ----  Fault flag on
!		if (OAdamp==1) then
!			cout = VOAdamp
!	        WRITE(*,*) 'OA damper is stuck'
!		endif
!		if (RAdamp==1) then
!			crec = VRAdamp
!	        WRITE(*,*) 'RA damper is stuck'
!		endif
!		if (EAdamp==1) then
!			cexh = VEAdamp
!	        WRITE(*,*) 'EA damper is stuck'
!		endif
! ----  Fault flag off
! **** Calculate resistance of each branch
! **** Resistance is sum of damper resistance and fixed resistance
!        rout = rdamper(cout,ropenout,fleakout,fareaout,alegg(iparout),&
!                       blegg(iparout),iparout) + rfixout
!        rrec = rdamper(crec,ropenrec,fleakrec,farearec,alegg(iparrec),&
!                       blegg(iparrec),iparrec) + rfixrec
!        rexh = rdamper(cexh,ropenexh,fleakexh,fareaexh,alegg(iparexh),&
!                       blegg(iparexh),iparexh) + rfixexh

        rout = rdamper(cout,ropenout,fleakout,fareaout,aleggoa(iparout),&
                       bleggoa(iparout),iparout) + rfixout
        rrec = rdamper(crec,ropenrec,fleakrec,farearec,aleggra(iparrec),&
                       bleggra(iparrec),iparrec) + rfixrec
        rexh = rdamper(cexh,ropenexh,fleakexh,fareaexh,aleggea(iparexh),&
                       bleggea(iparexh),iparexh) + rfixexh
! **** Calculate recirc and exhaust flow rates
! **** Call flow split routine
        call flowsplt(wext,pmix,pexh,0.0,rrec,rexh,wcritl,wcritu,rtolx,&
                      pext,wrec1,wexh1,ifail)
! ----  Fault flag on
		if (RAdamp==0) then
			wrec = wrec1
!	        WRITE(*,*) 'RA damper is fault free'
		elseif (RAdamp==1) then
			wrec = wrec1
!	        WRITE(*,*) 'RA damper is stuck'
		elseif (RAdamp==2) then
			if (crec==0) then
				wrec = VRAdamp
!				WRITE(*,*) 'RA damper is leaking'
			else
				wrec = wrec1     
			endif
		endif
		if (EAdamp==0) then
			wexh = wexh1
!	        WRITE(*,*) 'EA damper is fault free'
		elseif (EAdamp==1) then
			wexh = wexh1
!	        WRITE(*,*) 'EA damper is stuck'
		elseif (EAdamp==2) then
			if (cexh==0) then
				wexh = VEAdamp
!				WRITE(*,*) 'EA damper is leaking'
			else
				wexh = wexh1
			endif
		endif
! ----  Fault flag off
! **** Check for unsuccesful completion
        if (ifail==1) then
! **** One or more resistances negative
            stop 'type 325: resistances must not be negative'
        elseif (ifail==2) then
! **** Zero resistance for both outlet branches
            stop 'type 325: either rrec or rexh must be non-zero'
        endif
! **** Calculate outside air mass flow rate
        dp=pout-pmix
        wout1=wqudlin(rout,dp)
! ----  Fault flag on
		if (OAdamp==0) then
			wout = wout1
!	        WRITE(*,*) 'OA damper is fault free'
		elseif (OAdamp==1) then
			wout = wout1
!	        WRITE(*,*) 'OA damper is stuck'
		elseif (OAdamp==2) then
			if (cout==0) then
				wout = VOAdamp
!				WRITE(*,*) 'OA damper is leaking'
			else
				wout = wout1
			endif
		endif
! ----  Fault flag off
! **** Calculate mixed air flow rate as sum of outside and recirc flow rates
        wmix=wout+wrec
! **** Having calculated flows, calculate mixed temperature and humidity
! **** Model assumes wmix and wext both non-negative. Warn if not true.
        if (wmix<-1.0e-4 .and. lastit==1)&
            write(*,*) 'type 325: wmix < -1.0e-4'
        if (wext<-1.0e-4 .and. lastit==1)&
            write(*,*) 'type 325: wext < -1.0e-4'
! **** Check for reverse mixed flow
        if (wmix<=small) then
! **** Reverse mixed flow, take mixed temperature and humidity ratio to be
! **** an average of the outside and extract conditions so as to provide a
! **** reasonable guess for subsequent iterations
            tmix = (tout+text)/2.0
            gmix = (gout+gext)/2.0
        elseif (wmix>small .and. wext<=small) then
! **** Reverse extract flow - Mixed condition is the same as outside condition
            tmix = tout
            gmix = gout
        elseif (wmix>small .and. wext>small ) then
! **** Normal, forward mixed and extract flows - Calculate mixed conditions
! **** first calculate recirc air conditions if recirc flow positive
            if (wrec>small) then
! **** Test for reverse exhaust flow
                if (wexh<small) then
! **** Reverse exhaust flow - Recirc flow is a mixture of extract and outside
! **** conditions
                    wexhrev=-wexh
                    t(1) = text
                    gx(1) = gext
                    w(1) = wext
                    t(2) = tout
                    gx(2) = gout
                    w(2) = wexhrev
                    call moistmix(t,gx,w,2,trec,grec,wdummy)
                else
! **** Normal, forward exhaust flow - Recirc condition is extract condition
                    trec = text
                    grec = gext
                endif
            endif
! **** Calculate mixed air conditions
! **** Mixed condition is a combination of the recirc and outside conditions
! **** moistmix also tests for reverse flows and sets the mixed condition
! **** to the forward flow condition, or to an average of the flow conditions
! **** if both are reverse
            t(1) = trec
            gx(1) = grec
            w(1) = wrec
            t(2) = tout
            gx(2) = gout
            w(2) = wout
            call moistmix(t,gx,w,2,tmix,gmix,wdummy)
        else
            stop 'type325: logic fault'
        endif
! **** Assign outputs
        yout(1) = tmix
        yout(2) = gmix
        yout(3) = wmix
        yout(4) = pext
        yout(5) = wout
        yout(6) = wrec
        yout(7) = wexh
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo

        return
        end subroutine type325

! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Mixing box with minimum OA damper - implicit flow
! *
! * PURPOSE:    Calculates the mixed air temperature and humidity and
! *             the mixed air and extract air pressures. Uses Legg's
! *             correlations for the resistance of parallel and opposed
! *             blade dampers. Includes an additional damper in parallel
! *             with the main outside air damper.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tout    : Outside air dry bulb temperature                   (C)
! *  2. gout    : Outside air humidity ratio                     (kg/kg)
! *  3. text    : Extract air dry bulb temperature                   (C)
! *  4. gext    : Extract air humidity ratio                     (kg/kg)
! *  5. pout    : Outside air intake pressure                      (kPa)
! *  6. pexh    : Exhaust air outlet pressure                      (kPa)
! *  7. wmix    : Mixed air mass flow rate                        (kg/s)
! *  8. wext    : Extract air mass flow rate                      (kg/s)
! *  9. cout    : Outside air damper position (0=closed, 1=open)     (-)
! * 10. crec    : Recirc air damper position (0=open if PAR(16)=0)   (-)
! * 11. cexh    : Extract air damper position (0=closed, 1=open)     (-)
! * 12. cmou    : Minimum outside air damper pos. (0=closed, 1=open) (-)
! *
! * OUTPUTS
! * =======
! *  1. tmix    : Mixed air dry bulb temperature                     (C)
! *  2. gmix    : Mixed air humidity ratio                       (kg/kg)
! *  3. pmix    : Mixed air pressure                               (kPa)
! *  4. pext    : Extract air pressure                             (kPa)
! *  5. wout    : Outside air mass flow rate                      (kg/s)
! *  6. wrec    : Recirc air mass flow rate                       (kg/s)
! *  7. wexh    : Exhaust air mass flow rate                      (kg/s)
! *  8. wmou    : Mass flow rate thru minimum outside air damper  (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. iparout : Outside air damper: opposed (0) or parallel (1)    (-)
! *  2. iparrec : Recirc air damper: opposed (0) or parallel (1)     (-)
! *  3. iparexh : Exhaust air damper: opposed (0) or parallel (1)    (-)
! *  4. iparmou : Min outside air damper: opposed (0) or parallel (1)(-)
! *  5. ropenout: Open resist. of outside air damper         (0.001/k.m)
! *  6. ropenrec: Open resist. of recirc air damper          (0.001/k.m)
! *  7. ropenexh: Open resist. of exhaust air damper         (0.001/k.m)
! *  8. ropenmou: Open resist. of minimum outside air damper (0.001/k.m)
! *  9. fareaout: Face area of outside air damper                   (m2)
! * 10. farearec: Face area of recirc air damper                    (m2)
! * 11. fareaexh: Face area of exhaust air damper                   (m2)
! * 12. fareamou: Face area of minimum outside air damper           (m2)
! * 13. fleakout: Leakage for outside air damper (fraction of full flow)
! * 14. fleakrec: Leakage for recirc air damper  (fraction of full flow)
! * 15. fleakexh: Leakage for exhaust air damper (fraction of full flow)
! * 16. fleakmou: Leakage for minimum outside air damper      (fraction)
! * 17. rfixout : Fixed resistance in outside air branch     (0.001/k.m)
! * 18. rfixrec : Fixed resistance in recirc air branch      (0.001/k.m)
! * 19. rfixexh : Fixed resistance in exhaust air branch     (0.001/k.m)
! * 20. noninver: 0=invert recirc air damper, 1=not inverted         (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes positive mixed and extract flow rates
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 8, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  MOISTMIX
!   FUNCTIONS  CALLED:   None
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * routd   : resistance of main outside air damper
! * rmoud   : resistance of minimum outside air damper
! * routdp  : resistance of main and minimum dampers in parallel
! * rout    : total resistance of outside air branch
! * rrec    : total resistance of recirculation branch
! * rexh    : total resistance of exhaust branch
! * dpexhout: difference between exhaust and outside pressures
! * dpout   : pressure drop across outside branch
! * dprec   : pressure drop across recirculation branch
! * dpexh   : pressure drop across exhaust branch
! * srout   : "signed" resistance of outside branch
! * srrec   : "signed" resistance of recirculation branch
! * srexh   : "signed" resistance of exhaust branch
! * diff    : (b^2 - 4*a*c)/(2*a)
! * wxxxp   : flow in xxx branch corresponding to positive root
! * wxxxn   : flow in xxx branch corresponding to negative root
! * rootpos : true if positive root consistent with known directions
! * rootneg : true if negative root consistent with known directions
! * wsmall  : threshold for significant flow rate
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type326(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=12,no=8,np=20,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(20)    :: t,gx,w
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
        real         :: wsmall=1.0e-6
        logical      :: rootpos,rootneg
        real         :: fleakexh,fleakmou,rfixout ,rfixrec ,rfixexh ,&
                        crec,routd,rmoud,sqrroutd,sqrrmoud,routdp,&
                        rout,rrec,rexh,dpexhout,dpout,srout,dprec,&
                        srrec,dpexh,srexh,wout,wrec,wexh,aa,bb,cc,&
                        bbb,ccc,bbbsq,diff,z,wrecp,wrecn,wexhp,woutp,&
                        wexhn,woutn,pmix,pext,dpd,wmou,tmix,gmix,&
                        wexhrev,wdummy,grec,trec,tout,gout,text,gext,&
                        pout,pexh,wmix,wext,cout,cexh,cmou,ropenout,&
                        ropenrec,ropenexh,ropenmou,fareaout,farearec,&
                        fareaexh,fareamou,fleakout,fleakrec,rdamper,&
                        dpqudlin,wqudlin
        integer      :: i,noninver,iparout,iparrec,iparexh,iparmou

! **** Read in inputs
        tout = xin(1)
        gout = xin(2)
        text = xin(3)
        gext = xin(4)
        pout = xin(5)
        pexh = xin(6)
        wmix = xin(7)
        wext = xin(8)
        cout = xin(9)
        cexh = xin(11)
        cmou = xin(12)
! **** Read and check parameters
        iparout = nint(par_v(1))
        iparrec = nint(par_v(2))
        iparexh = nint(par_v(3))
        iparmou = nint(par_v(4))
        ropenout= par_v(5)
        ropenrec= par_v(6)
        ropenexh= par_v(7)
        ropenmou= par_v(8)
        fareaout= par_v(9)
        farearec= par_v(10)
        fareaexh= par_v(11)
        fareamou= par_v(12)
        fleakout= par_v(13)
        fleakrec= par_v(14)
        fleakexh= par_v(15)
        fleakmou= par_v(16)
        rfixout = par_v(17)
        rfixrec = par_v(18)
        rfixexh = par_v(19)
        noninver= nint(par_v(20))

        if (noninver/=1) then        !  moved 12/22/99
            crec = 1.0-xin(10)
        else
            crec = xin(10)
        endif

! **** Calculate resistance of each branch as sum of damper resistance and
! **** fixed resistance
! **** Outside air branch - Resistance of main and minimum dampers in
! **** parallel is in series with the fixed resistance
        routd = rdamper(cout,ropenout,fleakout,fareaout,alegg(iparout),&
                       blegg(iparout),iparout)
        rmoud = rdamper(cmou,ropenmou,fleakmou,fareamou,alegg(iparmou),&
                       blegg(iparmou),iparmou)
        sqrroutd = sqrt(routd)
        sqrrmoud = sqrt(rmoud)
        routdp =1./(1./routd + 1./rmoud + 2./(sqrroutd*sqrrmoud))
        rout  = routdp + rfixout
        if (rout<=0.0) stop &
            'type326: resistance in outside branch not greater than 0'
! **** Recirc and exhaust branches - one damper in series with fixed
! **** resistance
        rrec = rdamper(crec,ropenrec,fleakrec,farearec,alegg(iparrec),&
                       blegg(iparrec),iparrec) + rfixrec
        if (rrec<=0.0) stop &
            'type326: resistance in recirc branch not greater than 0'
        rexh = rdamper(cexh,ropenexh,fleakexh,fareaexh,alegg(iparexh),&
                       blegg(iparexh),iparexh) + rfixexh
        if (rexh<=0.0) stop &
            'type326: resistance in exhaust branch not greater than 0'
! **** Calculate "signed" resistances (see printed documentation)
! **** test for direction of the outside, return and exhaust flows by
! **** determining the sign of the pressure difference across the
! **** corresponding resistance assuming the flow in question is zero.
        dpexhout = pexh - pout
! **** Outside:
        dpout = rrec*wmix*abs(wmix) - rexh*(wext-wmix)*abs(wext-wmix) -&
                dpexhout
        srout = sign(rout,dpout)
! **** Return:
        dprec = rexh*wext*abs(wext) + rout*wmix*abs(wmix) + dpexhout
        srrec = sign(rrec,dprec)
! **** Exhaust:
        dpexh = rrec*wext*abs(wext) - rout*(wmix-wext)*abs(wmix-wext) -&
                dpexhout
        srexh = sign(rexh,dpexh)
! **** Determine necessity of solving quadratic - Check for special cases
        if (srout==0) then
! **** Outside flow zero
            wout = 0.0
            wrec = wmix
            wexh = wext - wrec
        elseif (srrec==0) then
! **** Recirculation flow zero
            wrec = 0.0
            wout = wmix
            wexh = wext
        elseif (srexh==0) then
! **** Exhaust flow zero
            wexh = 0.0
            wrec = wext
            wout = wmix -wrec
        else
! **** All three unknown flows non-zero:
! **** Calc wrec by solving a quadratic of the form a*wrec**2 + b*wrec + c = 0
! **** Note that aa=a, bb=-b/2, cc=c,  bbb=-b/2a,  ccc=c/a
            aa   = srout+srexh-srrec
            bb   = srout*wmix+srexh*wext
            cc   = srout*wmix*wmix+srexh*wext*wext+dpexhout
            if (aa==0.) then
! **** One root
                wrec = cc/(2.0*bb)
            else
! **** Two roots
                bbb = bb/aa
                ccc = cc/aa
                bbbsq = bbb*bbb
                diff = bbbsq-ccc
                if (diff>0) then
! **** Roots unequal
                    z = sqrt(diff)
                    wrecp = bbb + z
                    wrecn = bbb - z
! **** Check each root for consistency with directions of air flow
! **** already determined
                    wexhp = wext - wrecp
                    woutp = wmix - wrecp
                    if (wexhp*srexh<0.0 .or.&
                        wrecp*srrec<0.0 .or.&
                        woutp*srout<0.0) then
                        rootpos = .false.
                    else
                        rootpos = .true.
                    endif
                    wexhn = wext-wrecn
                    woutn = wmix-wrecn
                    if (wexhn*srexh<0.0 .or.&
                        wrecn*srrec<0.0 .or.&
                        woutn*srout<0.0) then
                        rootneg = .false.
                    else
                        rootneg = .true.
                    endif
                    if (.not.rootpos .and. .not.rootneg) then
                        stop 'type326: no consistent solution'
                    elseif (rootpos .and. rootneg) then
                           stop 'type326: ambiguous solution'
                       elseif (rootpos .and. .not.rootneg) then
                           wrec = wrecp
                       elseif (rootneg .and. .not.rootpos) then
                           wrec = wrecn
                    else
                        stop 'type326: logic fault 1'
                    endif
                elseif (diff==0) then
! **** Equal roots
                    wrec = bbb
                else
! **** Complex roots (coding error, negative resistances ...?)
!         write(*,*) (xin(iii), iii=1,ni)
!         write(*,*) (par(iii), iii=1,np)
!         write(*,*) rout, rrec, rexh
!         write(*,*) srout, srrec, srexh
!         write(*,*) bbbsq, ccc, diff
                    stop  'type 326: complex roots'
                endif
            endif
! **** Calculate remaining unknown flows
            wout = wmix - wrec
            wexh = wext - wrec
! **** Calculate mixed and extract pressures
            pmix = pout - srout*wout*wout
            pext = pexh + srexh*wexh*wexh
        endif
! **** Flow rate through minimum outside air damper
        dpd  = dpqudlin(routdp,wout)
        wmou = wqudlin(rmoud,dpd)
! **** Having calculated flows, calculate mixed temperature and humidity
! **** Model assumes wmix and wext both non-negative. Warn if not true.
        if (lastit==1) then
            if (wmix<-wsmall) write(*,*) 'type 326: wmix < ',-wsmall
            if (wext<-wsmall) write(*,*) 'type 326: wext < ',-wsmall
        endif
! **** Check for zero or reverse mixed flow
        if (wmix<=wsmall) then
! **** Reverse mixed flow, take mixed temperature and humidity ratio to be
! **** an average of the outside and extract conditions so as to provide a
! **** reasonable guess for subsequent iterations
            tmix = (tout+text)/2.0
            gmix = (gout+gext)/2.0
        elseif (wmix>wsmall .and. wext<=wsmall) then
! **** Zero/reverse extract flow - Mixed condition is the same as outside
! **** condition
            tmix = tout
            gmix = gout
        elseif (wmix>wsmall .and. wext>wsmall ) then
! **** Normal, forward mixed and extract flows - Calculate mixed conditions
! **** first calculate recirc air conditions if recirc flow positive
            if (wrec>wsmall) then
! **** Test for zero/reverse exhaust flow
                if (wexh<wsmall) then
! **** Reverse exhaust flow - Recirc flow is a mixture of extract and outside
! **** conditions
                    wexhrev=-wexh
                    t(1) = text
                    gx(1) = gext
                    w(1) = wext
                    t(2) = tout
                    gx(2) = gout
                    w(2) = wexhrev
                    call moistmix(t,gx,w,2,trec,grec,wdummy)
                else
! **** Normal, forward exhaust flow - Recirc condition is extract condition
                    trec = text
                    grec = gext
                endif
            endif
! **** Calculate mixed air conditions
! **** Mixed condition is a combination of the recirc and outside conditions
! **** moistmix also tests for reverse flows and sets the mixed condition
! **** to the forward flow condition, or to an average of the flow conditions
! **** if both are reverse
            t(1) = trec
            gx(1) = grec
            w(1) = wrec
            t(2) = tout
            gx(2) = gout
            w(2) = wout
            call moistmix(t,gx,w,2,tmix,gmix,wdummy)
        else
            stop 'type326: logic fault 2'
        endif
! **** Assign outputs
        yout(1) = tmix
        yout(2) = gmix
        yout(3) = pmix
        yout(4) = pext
        yout(5) = wout
        yout(6) = wrec
        yout(7) = wexh
        yout(8) = wmou
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo

        return
        end subroutine type326

! *********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE: Mixing box with minimum outside air damper
! *
! * PURPOSE:    Calculates the mixed air flow rate, temperature and
! *             humidity and the extract air pressure. Uses Legg's
! *             correlations for the resistance of parallel and opposed
! *             blade dampers. Includes an additional damper in parallel
! *             with the main outside air damper.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tri     : Outside air dry bulb temperature                   (C)
! *  2. gout    : Outside air humidity ratio                     (kg/kg)
! *  3. text    : Extract air dry bulb temperature                   (C)
! *  4. gext    : Extract air humidity ratio                     (kg/kg)
! *  5. pout    : Outside air intake gauge pressure                (kPa)
! *  6. pexh    : Exhaust air outlet gauge pressure                (kPa)
! *  7. pmix    : Mixed air gauge pressure                         (kPa)
! *  8. wext    : Extract dry air mass flow rate                  (kg/s)
! *  9. cout    : Outside air damper position (0=closed, 1=open)     (-)
! * 10. cre     : Recirc air damper position (0=open if PAR(16)=0)   (-)
! * 11. cexh    : Extract air damper position (0=closed, 1=open)     (-)
! * 12. cmou    : Minimum outside air damper pos. (0=closed,1=open)  (-)
! *
! * OUTPUTS
! * =======
! *  1. tmix    : Mixed air dry bulb temperature                     (C)
! *  2. gmix    : Mixed air humidity ratio                       (kg/kg)
! *  3. wmix    : Mixed air mass flow rate                        (kg/s)
! *  4. pext    : Extract air gauge pressure                       (kPa)
! *  5. wout    : Total outside air mass flow rate                (kg/s)
! *  6. wrec    : Recirc air mass flow rate                       (kg/s)
! *  7. wexh    : Exhaust air mass flow rate                      (kg/s)
! *  8. wmou    : Mass flow rate thru minimum outside air damper  (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. iparout : Outside air damper: opposed (0) or parallel (1)    (-)
! *  2. iparrec : Recirc air damper: opposed (0) or parallel (1)     (-)
! *  3. iparexh : Exhaust air damper: opposed (0) or parallel (1)    (-)
! *  4. iparmou : Min outside air damper: opposed (0) or parallel (1)(-)
! *  5. ropenout: Open resist. of outside air damper         (0.001/k.m)
! *  6. ropenrec: Open resist. of recirc air damper          (0.001/k.m)
! *  7. ropenexh: Open resist. of exhaust air damper         (0.001/k.m)
! *  8. ropenmou: Open resist. of minimum outside air damper (0.001/k.m)
! *  9. fareaout: Face area of outside air damper                   (m2)
! * 10. farearec: Face area of recirc air damper                    (m2)
! * 11. fareaexh: Face area of exhaust air damper                   (m2)
! * 12. fareamou: Face area of minimum outside air damper           (m2)
! * 13. fleakout: Leakage for outside air damper (fraction of full flow)
! * 14. fleakrec: Leakage for recirc air damper  (fraction of full flow)
! * 15. fleakexh: Leakage for exhaust air damper (fraction of full flow)
! * 16. fleakmou: Leakage for minimum outside air damper      (fraction)
! * 17. rfixout :  Fixed resistance in outside air branch    (0.001/k.m)
! * 18. rfixrec :  Fixed resistance in recirc air branch     (0.001/k.m)
! * 19. rfixexh :  Fixed resistance in exhaust air branch    (0.001/k.m)
! * 20. noninver: 0=invert recirc air damper, 1=not inverted         (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes positive mixed and extract flow rates
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 29, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWSPLT, MOISTMIX
!   FUNCTIONS  CALLED:   None
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:
! 
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * routd   : resistance of main outside air damper
! * rmoud   : resistance of minimum outside air damper
! * routdp  : resistance of main and minimum dampers in parallel
! * rout    : total resistance of outside air branch
! * rrec    : total resistance of recirculation branch
! * rexh    : total resistance of exhaust branch
! * dp      : pressure drop across outside air branch
! * dpd     : pressure drop across outside air dampers
! * small   : threshold for significant flow rate
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type327(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=12,no=8,np=20,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real,dimension(20)    :: t,gx,w
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
        real         :: small=1.0e-10,wcritl=3.e-2,wcritu=6.e-2
        real         :: ropenexh,ropenmou,fareaout,farearec,fareaexh,&
                        fareamou,fleakout,fleakrec,fleakexh,fleakmou,&
                        rfixout,rfixrec,rfixexh,tout,gout,text,&
                        gext,pout,pexh,pmix,wext,cout,crec,cexh,cmou,&
                        routd,rmoud,sqrroutd,sqrrmoud,routdp,rout,rrec,&
                        rexh,wexh,wrec,pext,dp,wout,dpd,wmou,wmix,tmix,&
                        gmix,wexhrev,wdummy,grec,trec,ropenout,ropenrec,&
                        rdamper,wqudlin,dpqudlin
        integer      :: i,noninver,ifail,iparout,iparrec,iparexh,iparmou

! **** Read and check parameters
        iparout = nint(par_v(1))
        iparrec = nint(par_v(2))
        iparexh = nint(par_v(3))
        iparmou = nint(par_v(4))
        ropenout= par_v(5)
        ropenrec= par_v(6)
        ropenexh= par_v(7)
        ropenmou= par_v(8)
        fareaout= par_v(9)
        farearec= par_v(10)
        fareaexh= par_v(11)
        fareamou= par_v(12)
        fleakout= par_v(13)
        fleakrec= par_v(14)
        fleakexh= par_v(15)
        fleakmou= par_v(16)
        rfixout = par_v(17)
        rfixrec = par_v(18)
        rfixexh = par_v(19)
        noninver= nint(par_v(20))
! **** Read in inputs
        tout = xin(1)
        gout = xin(2)
        text = xin(3)
        gext = xin(4)
        pout = xin(5)
        pexh = xin(6)
        pmix = xin(7)
        wext = xin(8)
        cout = xin(9)
        if (noninver/=1) then
            crec = 1.0-xin(10)
        else
            crec = xin(10)
        endif
        cexh = xin(11)
        cmou = xin(12)
! **** Calculate resistance of each branch as sum of damper resistance and
! **** fixed resistance
! **** Outside air branch - Resistance of main and minimum dampers in
! **** parallel is in series with the fixed resistance
        routd = rdamper(cout,ropenout,fleakout,fareaout,alegg(iparout),&
                       blegg(iparout),iparout)
        rmoud  = rdamper(cmou,ropenmou,fleakmou,fareamou,alegg(iparmou),&
                       blegg(iparmou),iparmou)
        sqrroutd = sqrt(routd)
        sqrrmoud = sqrt(rmoud)
        routdp =1./(1./routd + 1./rmoud + 2./(sqrroutd*sqrrmoud))
        rout  = routdp + rfixout
        if (rout<=0.0) stop &
            'type327: resistance in outside branch not greater than 0'
! **** Recirc and exhaust branches - One damper in series with fixed
! **** resistance
        rrec  = rdamper(crec,ropenrec,fleakrec,farearec,alegg(iparrec),&
                        blegg(iparrec),iparrec) + rfixrec
        if (rrec<=0.0) stop &
            'type327: resistance in recirc branch not greater than 0'
        rexh  = rdamper(cexh,ropenexh,fleakexh,fareaexh,alegg(iparexh),&
                        blegg(iparexh),iparexh) + rfixexh
        if (rexh<=0.0) stop &
            'type327: resistance in exhaust branch not greater than 0'
! **** Calculate recirc and exhaust flow rates
! **** Call flow split routine
        call flowsplt(wext,pmix,pexh,0.0,rrec,rexh,wcritl,wcritu,rtolx,&
                      pext,wrec,wexh,ifail)
! **** Check for unsuccesful completion
        if (ifail==1) then
! **** One or more resistances negative
            stop 'type 327: resistances must not be negative'
        elseif (ifail==2) then
! **** Zero resistance for both outlet branches
            stop 'type 327: either rret or rexh must be non-zero'
        endif
! **** Total outside air mass flow rate
        dp   = pout-pmix
        wout = wqudlin(rout,dp)
! **** Flow rate through minimum outside air damper
        dpd  = dpqudlin(routdp,wout)
        wmou = wqudlin(rmoud,dpd)
! **** Calculate mixed air flow rate as sum of outside and recirc flow rates
        wmix = wout+wrec
! **** Having calculated flows, calculate mixed temperature and humidity
! **** Model assumes wmix and wext both non-negative. Warn if not true.
        if (wmix<-1.0e-4 .and. lastit==1)&
            write(*,*) 'type 327: wmix < -1.0e-4'
        if (wext<-1.0e-4 .and. lastit==1)&
            write(*,*) 'type 327: wext < -1.0e-4'
! **** Check for reverse mixed flow
        if (wmix<=small) then
! **** Reverse mixed flow, take mixed temperature and humidity ratio to be
! **** an average of the outside and extract conditions so as to provide a
! **** reasonable guess for subsequent iterations
            tmix = (tout+text)/2.0
            gmix = (gout+gext)/2.0
        elseif (wmix>small .and. wext<=small) then
! **** Reverse extract flow - Mixed condition is the same as outside condition
            tmix = tout
            gmix = gout
        elseif (wmix>small .and. wext>small ) then
! **** Normal, forward mixed and extract flows - Calculate mixed conditions
! **** first calculate recirc air conditions if recirc flow positive
            if (wrec>small) then
! **** Test for reverse exhaust flow
                if (wexh<small) then
! **** Reverse exhaust flow - Recirc flow is a mixture of extract and outside
! **** conditions
                    wexhrev = -wexh
                    wexhrev = -wexh
                    t(1) = text
                    gx(1) = gext
                    w(1) = wext
                    t(2) = tout
                    gx(2) = gout
                    w(2) = wexhrev
                    call moistmix(t,gx,w,2,trec,grec,wdummy)
                else
! **** Normal, forward exhaust flow - Recirc condition is extract condition
                    trec = text
                    grec = gext
                endif
            endif
! **** Calculate mixed air conditions
! **** Mixed condition is a combination of the recirc and outside conditions
! **** moistmix also tests for reverse flows and sets the mixed condition
! **** to the forward flow condition, or to an average of the flow conditions
! **** if both are reverse
            t(1) = trec
            gx(1) = grec
            w(1) = wrec
            t(2) = tout
            gx(2) = gout
            w(2) = wout
            call moistmix(t,gx,w,2,tmix,gmix,wdummy)
        else
            stop 'type327: logic fault'
        endif
! **** Assign outputs
        yout(1) = tmix
        yout(2) = gmix
        yout(3) = wmix
        yout(4) = pext
        yout(5) = wout
        yout(6) = wrec
        yout(7) = wexh
        yout(8) = wmou
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo

        return
        end subroutine type327

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Two port control valve
! *
! * PURPOSE:     Calculates the inlet pressure from the flow rate and
! *              outlet pressure
! **********************************************************************
! * INPUTS
! * ======
! *  1. ww      : water mass flow rate                            (kg/s)
! *  2. pwo     : outlet water pressure                            (kPa)
! *  3. vstem   : valve stem position                                (-)
! *
! * OUTPUTS
! * =======
! *  1. pwi     : inlet water pressure                             (kPa)
! *
! * PARAMETERS
! * ==========
! *  1. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! *  2. eqpchar : valve curvature parameter (0=linear)               (-)
! *  3. xeqp    : equal percentage/linear break point                (-)
! *  4. sv      : valve rangability                                  (-)
! *  5. cl      : valve leakage (fractional flow)                    (-)
! *  6. xlin    : linear/close-off break point                       (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes characteristic has three segments -
!                        close-off, linear and exponential. Linear
!                        characteristic obtained by setting linear/
!                        exponential break point to 1. Close-off
!                        segment is also linear.
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 14, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RLINPORT, REQPPORT
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           (Based on confidential information from a major
!                        manufacturer.)
! 
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type328(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=3,no=1,np=6,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kv
! **** Minimum water-side resistance
        real         :: frsmall=1.0e-8

        real         :: ww,pwo,vstem,eqpchar,xeqp,sv,cl,xlin,frwvalve,&
                        rlinport,reqpport,pwi,dpqudlin

! **** Read in inputs
        ww      = xin(1)
        pwo     = xin(2)
        vstem   = xin(3)
! **** Read in parameters
        kv      = par_v(1)
        eqpchar = par_v(2)
        xeqp    = par_v(3)
        sv      = par_v(4)
        cl      = par_v(5)
        xlin    = par_v(6)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
            frwvalve=rlinport(vstem,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frwvalve=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
        if (frwvalve<frsmall) then
! **** Total resistance almost zero
            write(*,*) 'type328: resistance must not be < ',frsmall
            stop
        endif
! **** Calculate water inlet pressure from flow resistance
        pwi     = pwo+dpqudlin(frwvalve,ww)
! **** Assign output
        yout(1)  = pwi
! **** Allow freezing of algebraic variable
        iostat(1)=1

        return
        end subroutine type328
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Two port control valve
! *
! * PURPOSE:     Calculates the flow rate from the inlet pressure and
! *              outlet pressure
! **********************************************************************
! * INPUTS
! * ======
! *  1. pwi     : inlet water gauge pressure                       (kPa)
! *  2. pwo     : outlet water gauge pressure                      (kPa)
! *  3. vstem   : valve stem position                                (-)
! *
! * OUTPUTS
! * =======
! *  1. ww      : water mass flow rate                            (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! *  2. eqpchar : valve curvature parameter (0=linear)               (-)
! *  3. xeqp    : equal percentage/linear break point                (-)
! *  4. sv      : valve rangability                                  (-)
! *  5. cl      : valve leakage (fractional flow)                    (-)
! *  6. xlin    : linear/close-off break point                       (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Assumes characteristic has three segments -
!                        close-off, linear and exponential. Linear
!                        characteristic obtained by setting linear/
!                        exponential break point to 1. Close-off
!                        segment is also linear.
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 14, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RLINPORT, REQPPORT
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           (Based on confidential information from a major
!                        manufacturer.)
! 
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type329(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=3,no=1,np=6,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kv
! **** Minimum water-side resistance,
        real         :: frsmall=1.0e-8
        real         :: pwi,pwo,vstem,eqpchar,xeqp,sv,cl,xlin,&
                        frwvalve,rlinport,reqpport,dp,ww,wqudlin

! **** Read in inputs
        pwi     = xin(1)
        pwo     = xin(2)
        vstem   = xin(3)
! **** Read in parameters
        kv      = par_v(1)
        eqpchar = par_v(2)
        xeqp    = par_v(3)
        sv      = par_v(4)
        cl      = par_v(5)
        xlin    = par_v(6)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
            frwvalve=rlinport(vstem,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frwvalve=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
        if (frwvalve<frsmall) then
! **** Resistance almost zero
            write(*,*) 'type329: resistance must not be < ',frsmall
            stop
        endif
! **** Calculate water inlet pressure from flow resistance
        dp      = pwi-pwo
        ww      = wqudlin(frwvalve,dp)
! **** Assign output
        yout(1)  = ww
! **** Allow freezing of algebraic variable
        iostat(1)=1
        return
        end subroutine type329
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Three port mixing valve
! *
! * PURPOSE:     Calculates the pressure at each inlet port and the flow
! *              at the common port from the flows at each inlet port
! *              and the outlet pressure.
! **********************************************************************
! * INPUTS
! * ======
! *  1. wwflow  : flow rate through flow port                     (kg/s)
! *  2. wwbypas : flow rate through by-pass port                  (kg/s)
! *  3. pwo     : outlet water pressure                            (kPa)
! *  4. vstem   : valve stem position                                (-)
! *
! * OUTPUTS
! * =======
! *  1. pwflow  : inlet pressure at flow port                      (kPa)
! *  2. pwbypas : inlet pressure at by-pass port                   (kPa)
! *  3. wwo     : outlet water flow rate                          (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. ivaltype: valve type: 0=lin/lin, 1=eq%(flow)/lin(byp), 2=lin/eq%
! *  2. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! *  3. eqpchar : valve curvature parameter (equal percentage port)  (-)
! *  4. xeqp    : equal percentage/linear break point                (-)
! *  5. sv      : valve rangability                                  (-)
! *  6. cl      : valve leakage (fractional flow)                    (-)
! *  7. xlin    : linear/close-off break point                       (-)
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                November 14, 1995
! 
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           (Based on confidential information from a
!                        major manufacturer.)
! 
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type330(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=4,no=3,np=7,ns=0
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kv
! **** Minimum water-side resistance,
        real         :: frsmall=1.0e-8

        real         :: wwflow,wwbypas,pwo,vstem,eqpchar,xeqp,sv,cl,&
                        xlin,vstembyp,frvalflo,rlinport,reqpport,&
                        frvalbyp,pwflow,dpqudlin,pwbypas,wwo
        integer      :: i,ivaltype

! **** Read in inputs
        wwflow  = xin(1)
        wwbypas = xin(2)
        pwo     = xin(3)
        vstem   = xin(4)
! **** Read in parameters
        ivaltype= nint(par_v(1))
        kv      = par_v(2)
        eqpchar = par_v(3)
        xeqp    = par_v(4)
        sv      = par_v(5)
        cl      = par_v(6)
        xlin    = par_v(7)
! **** Valve types and resistances
! **** limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
        vstembyp= 1.-vstem
! **** Flow port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==2) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalflo=rlinport(vstem,kv,sv,xlin,cl)
        elseif (ivaltype==1) then
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalflo=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        else
            stop 'type330: illegal valve type'
        endif
! **** By-pass port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==1) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalbyp=rlinport(vstembyp,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalbyp=reqpport(vstembyp,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
        if (frvalflo<frsmall .or. frvalbyp<frsmall) then
! **** Total resistance almost zero
            write(*,*) &
            'type330: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Pressure at inlet to flow and by-pass ports
        pwflow  = pwo+dpqudlin(frvalflo,wwflow)
        pwbypas = pwo+dpqudlin(frvalbyp,wwbypas)
! **** Outlet flow rate
        wwo     = wwflow+wwbypas
! **** Assign output values
        yout(1)  = pwflow
        yout(2)  = pwbypas
        yout(3)  = wwo
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i)=1
        enddo
        return
        end subroutine type330
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Variable speed drive
! *
! * PURPOSE:        Calculate the rotation speed of an ideal motor
! *                 connected to a variable speed drive incorporating a
! *                 rate limit.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. fspddem : demanded fractional speed                          (-)
! *
! * OUTPUTS
! * =======
! *  1. rspd    : actual rotation speed                          (rev/s)
! *
! * PARAMETERS
! * ==========
! *  1. rspdmax : maximum rotation speed                         (rev/s)
! *  2. ttran   : travel time (lim-lim)                              (s)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. fspdact : actual fractional speed at previous call
! *  3. fspdact : actual fractional speed at previous timestep
! *
! **********************************************************************
! 
!   MAJOR RESTRICTIONS:  Neglects motor dynamics and drive hysteresis
!                        and non-linearity. Does not treat power
!                        consumption or efficiency.
! 
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
! 
!   DATE:                June 16, 1994
! 
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:
! 
!   REVISION HISTORY:    None
! 
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! * fspdactp: actual fractional speed at the previous step time
! * dfspdmax: maximum change in fractional speed in one time-step
! * dfspd   : actual change in fractional speed during time-step
! * fspdact : current actual fractional speed
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type333(xin,yout,par_v,saved_v,iostat)
 
        use modsim_head
!        implicit none
        integer,parameter                 :: ni=1,no=1,np=2,ns=3
        real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        logical      :: quick
        real         :: small=1.e-6
        
! **** Read in inputs
        fspddem  = xin(1)
! **** Read in parameters
        rspdmax  = par_v(1)
        ttran    = par_v(2)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                do is = 2,ns-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - Update previous sample instant values
            do is=2,ns-1,2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        fspdactp = saved_v(3)
! **** Limit demanded fractional speed
        fspddem = max(0.,min(fspddem,1.))
! **** determine if demanded position attained
        if (ttran<=(tstep*small)) then
! **** Response is essentially instantaneous
            quick=.true.
        else
            quick=.false.
! **** Travel time is non-zero - Determine maximum change in fractional
! **** speed in one time-step
            dfspdmax=tstep/ttran
        endif
! **** Difference between current demanded speed and previous actual speed
        dfspd=fspddem-fspdactp
        if (quick .or. abs(dfspd)<=dfspdmax) then
! **** Demanded speed attained
            fspdact=fspddem
        else
! **** Demanded speed not attained
            fspdact=fspdactp+sign(dfspdmax,dfspd)
        endif
! **** Actual rotation speed
        rspd=fspdact*rspdmax
! **** Save time of current call
        saved_v(1)=time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)=fspdact
! **** Output
        yout(1)=rspd
! **** Disallow freezing
        do i=1,no
            iostat(i)=0
        enddo
! **** Return
        return
        end subroutine type333
        
